/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2022-2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::generatedCellZone

Description
    cellZone selection or generation class

    The cellZone is either looked-up from the mesh or generated by a
    zoneGenerator and stored locally.

Usage
    Examples:
    \verbatim
        // Select all the cells
        cellZone all;

        // Select the cells within the cellZone rotor
        cellZone rotor;

        // Select the cells within the given box
        cellZone
        {
            type        box;
            box         (3.48 -0.1 -0.1) (3.5 0.1 0.1);
        }

        // Select the cell containing the given point
        cellZone
        {
            type            containsPoints;
            points          ((0.075 0.2 0.05));
        }
    \endverbatim

See also
    Foam::generatedZoneSet
    Foam::zoneGenerator

SourceFiles
    generatedCellZone.C
    generatedCellZoneI.H

\*---------------------------------------------------------------------------*/

#ifndef generatedCellZone_H
#define generatedCellZone_H

#include "generatedZoneSet.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class polyMesh;
class polyMeshMap;
class polyDistributionMap;

/*---------------------------------------------------------------------------*\
                        Class generatedCellZone Declaration
\*---------------------------------------------------------------------------*/

class generatedCellZone
{
    // Private data

        //- Reference to the mesh
        const polyMesh& mesh_;

        //- Switch for all cells
        bool all_;

        generatedZoneSet cellZone_;


public:

    // Constructors

        //- Construct from mesh selecting all cells
        generatedCellZone(const polyMesh& mesh);

        //- Construct from mesh and dictionary
        generatedCellZone(const polyMesh& mesh, const dictionary& dict);

        //- Disallow default bitwise copy construction
        generatedCellZone(const generatedCellZone&) = delete;


    //- Destructor
    ~generatedCellZone();


    // Member Functions

        // Access

            //- Return true if the set comprises all the cells
            inline bool all() const;

            //- Return const access to the name of cellZone
            inline const word& name() const;

            //- Return const access to the cell set
            inline const cellZone& zone() const;

            //- Return the number of cells in the set
            inline label nCells() const;

            //- Return the cell index corresponding to the cell set index
            inline label celli(const label i) const;


        // Mesh changes

            //- Update for mesh motion
            void movePoints();

            //- Update topology using the given map
            void topoChange(const polyTopoChangeMap&);

            //- Update from another mesh using the given map
            void mapMesh(const polyMeshMap&);

            //- Redistribute or update using the given distribution map
            void distribute(const polyDistributionMap&);


        // IO

            //- Read coefficients dictionary
            bool read(const dictionary& dict);


    // Member Operators

        //- Disallow default bitwise assignment
        void operator=(const generatedCellZone&) = delete;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#include "generatedCellZoneI.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
