/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     | Website:  https://openfoam.org
    \\  /    A nd           | Copyright (C) 2025 OpenFOAM Foundation
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

Class
    Foam::zoneGenerators::insideSurface

Description
    A zoneGenerator which selects points, cells or faces with centres either
    inside or outside a surface

    By default all the points, cells or faces of the mesh are tested for being
    inside or outside the surface but an optional single zone or list of zones
    may be provided and those points, cells or faces are tested instead.  This
    provides an efficient method of hierarchical sub-division of space where an
    initial selection of points cells or faces is refined by selecting the
    sub-set inside or outside the given surface rather than having to generate
    another zone to intersect with.

Usage
    \table
        Property     | Description             | Required  | Default value
        type         | Type: insideSurface     | yes       |
        name         | Name of the zone        | no        | zoneGenerator name
        zoneType     | Type of zone            | yes       |
        select       | Select either the inside or outside | no | inside
        moveUpdate   | Switch to update after mesh motion  | no | false
        surface      | searchableSurface type  | yes       |
        zone         | Optional zone to sub-set            | no  |
        zones        | Optional list of zones to sub-set   | no  |
    \endtable

    A cellZone named \c cylinder containing the cells with centres inside the
    \c closedTriSurface "sloshingCylinder.obj" is generated by
    \verbatim
        cylinder
        {
            type        insideSurface;
            zoneType    cell;

            surface     closedTriSurface;
            file        "sloshingCylinder.obj";
        }
    \endverbatim

See also
    zoneGenerators::volume

SourceFiles
    insideSurface.C

\*---------------------------------------------------------------------------*/

#ifndef insideSurface_zoneGenerator_H
#define insideSurface_zoneGenerator_H

#include "volume.H"

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

namespace Foam
{

class searchableSurface;

namespace zoneGenerators
{

/*---------------------------------------------------------------------------*\
                              Class insideSurface Declaration
\*---------------------------------------------------------------------------*/

class insideSurface
:
    public volume
{
    // Private Data

        //- Surface
        autoPtr<searchableSurface> surfacePtr_;


protected:

    // Protected Member Functions

        friend class volume;

        template<class ZoneType, class UnaryOp>
        labelList select
        (
            const insideSurface& zoneGen,
            const vectorField& pts,
            const UnaryOp& uop
        ) const;

        template<class ZoneType, class UnaryOp>
        labelList select
        (
            const insideSurface& zoneGen,
            const zoneGeneratorList& zoneGenerators,
            const vectorField& pts,
            const UnaryOp& uop
        ) const;

        template<class UnaryOp>
        labelList select
        (
            const insideSurface& zoneGen,
            const zoneGeneratorList& zoneGenerators,
            const vectorField& pts,
            boolList& flipMap,
            const UnaryOp& uop
        ) const;


public:

    //- Runtime type information
    TypeName("insideSurface");


    // Constructors

        //- Construct from dictionary
        insideSurface
        (
            const word& name,
            const polyMesh& mesh,
            const dictionary& dict
        );


    //- Destructor
    virtual ~insideSurface();


    // Member Functions

        virtual zoneSet generate() const;
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace zoneGenerators
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
