/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2016 OpenFOAM Foundation
    Copyright (C) 2016-2025 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

template<class T, int SizeMin>
template<class ListType>
inline void Foam::DynamicField<T, SizeMin>::doAssignDynList
(
    const ListType& list
)
{
    const label len = list.size();

    if (capacity_ < len)
    {
        // Needs more space for the copy operation
        List<T>::setAddressableSize(capacity_);
        List<T>::resize_nocopy(len);
        capacity_ = List<T>::size();
    }

    // Perform copy into addressable portion
    List<T>::setAddressableSize(len);
    List<T>::operator=(list);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::doCapacity_copy
(
    label count,
    const label newCapacity
)
{
    if (newCapacity == capacity_)
    {
        return;
    }

    // Addressable length, possibly truncated by new capacity
    const label currLen = std::min(List<T>::size(), newCapacity);

    // The count truncated by the new addressable range
    count = std::min(count, currLen);

    // Consistent allocated sizing
    List<T>::setAddressableSize(capacity_);

    if (count > 0)
    {
        List<T>::resize_copy(count, newCapacity);
    }
    else
    {
        List<T>::resize_nocopy(newCapacity);
    }

    capacity_ = List<T>::size();
    List<T>::setAddressableSize(currLen);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::doReserve_copy
(
    label count,
    const label len
)
{
    if (capacity_ < len)
    {
        // Preserve addressed size
        const label currLen = List<T>::size();

        // The count truncated by the new addressable range
        count = std::min(count, currLen);

        // Consistent allocated sizing
        List<T>::setAddressableSize(capacity_);

        // Increase capacity (eg, doubling)
        capacity_ =
            Foam::ListPolicy::reserve_size<SizeMin, 2>(len, capacity_);

        if (count > 0)
        {
            List<T>::resize_copy(count, capacity_);
        }
        else
        {
            List<T>::resize_nocopy(capacity_);
        }

        capacity_ = List<T>::size();
        List<T>::setAddressableSize(currLen);
    }
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class T, int SizeMin>
inline constexpr Foam::DynamicField<T, SizeMin>::DynamicField() noexcept
:
    Field<T>(),
    capacity_(0)
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField(const label initialCapacity)
:
    Field<T>(),
    capacity_(0)
{
    reserve_nocopy(initialCapacity);
}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    std::pair<label,label> sizing
)
:
    Field<T>(std::max(sizing.first, sizing.second)),
    capacity_(Field<T>::size())
{
    List<T>::setAddressableSize(sizing.first);
}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const label len,
    const T& val
)
:
    Field<T>(len, val),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const label len,
    Foam::zero
)
:
    Field<T>(len, Foam::zero{}),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const DynamicField<T, SizeMin>& list
)
:
    Field<T>(list),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
template<int AnySizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const DynamicField<T, AnySizeMin>& list
)
:
    Field<T>(list),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const UList<T>& list
)
:
    Field<T>(list),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
template<class Addr>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const IndirectListBase<T, Addr>& list
)
:
    Field<T>(list),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    List<T>&& content
) noexcept
:
    Field<T>(std::move(content)),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
template<int AnySizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    DynamicList<T, AnySizeMin>&& content
) noexcept
:
    Field<T>(std::move(static_cast<List<T>&>(content))),
    capacity_(content.capacity())
{
    content.setCapacity_unsafe(0);  // All contents moved
}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    DynamicField<T, SizeMin>&& content
) noexcept
:
    Field<T>(std::move(static_cast<List<T>&>(content))),
    capacity_(content.capacity())
{
    content.setCapacity_unsafe(0);  // All contents moved
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    DynamicField<T, AnySizeMin>&& content
) noexcept
:
    Field<T>(std::move(static_cast<List<T>&>(content))),
    capacity_(content.capacity())
{
    content.setCapacity_unsafe(0);  // All contents moved
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    DynamicField<T, AnySizeMin>& content,
    bool reuse
)
:
    Field<T>(),
    capacity_(0)
{
    if (reuse)
    {
        Field<T>::transfer(static_cast<List<T>&>(content));
        capacity_ = content.capacity();
        content.setCapacity_unsafe(0);  // All contents moved
    }
    else
    {
        Field<T>::operator=(content);
        capacity_ = content.size();
    }
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    DynamicList<T, AnySizeMin>& content,
    bool reuse
)
:
    Field<T>(),
    capacity_(0)
{
    if (reuse)
    {
        Field<T>::transfer(static_cast<List<T>&>(content));
        capacity_ = content.capacity();
        content.setCapacity_unsafe(0);  // All contents moved
    }
    else
    {
        Field<T>::operator=(content);
        capacity_ = content.size();
    }
}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    List<T>& content,
    bool reuse
)
:
    Field<T>(),
    capacity_(0)
{
    if (reuse)
    {
        Field<T>::transfer(content);
    }
    else
    {
        Field<T>::operator=(content);
    }
    capacity_ = Field<T>::size();
}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const UList<T>& mapF,
    const labelUList& mapAddressing
)
:
    Field<T>(mapF, mapAddressing),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const UList<T>& mapF,
    const labelListList& mapAddressing,
    const scalarListList& weights
)
:
    Field<T>(mapF, mapAddressing, weights),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField
(
    const UList<T>& mapF,
    const FieldMapper& map
)
:
    Field<T>(mapF, map),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::DynamicField<T, SizeMin>::DynamicField(Istream& is)
:
    Field<T>(is),
    capacity_(Field<T>::size())
{}


template<class T, int SizeMin>
inline Foam::tmp<Foam::DynamicField<T, SizeMin>>
Foam::DynamicField<T, SizeMin>::clone() const
{
    return tmp<DynamicField<T, SizeMin>>::New(*this);
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class T, int SizeMin>
inline std::streamsize
Foam::DynamicField<T, SizeMin>::capacity_bytes() const noexcept
{
    return std::streamsize(capacity_)*sizeof(T);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::setCapacity
(
    const label len
)
{
    // Preserve all content
    doCapacity_copy(List<T>::size(), len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::setCapacity_nocopy
(
    const label len
)
{
    // Preserve 0 content
    doCapacity_copy(0, len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::reserve
(
    const label len
)
{
    // Preserve all content
    doReserve_copy(List<T>::size(), len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::reserve_nocopy
(
    const label len
)
{
    // Preserve 0 content
    doReserve_copy(0, len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::reserve_exact
(
    const label len
)
{
    if (capacity_ < len)
    {
        // Preserve addressed size
        const label currLen = List<T>::size();

        // Consistent allocated sizing
        List<T>::setAddressableSize(capacity_);

        // Exact length
        capacity_ = len;

        // if (!nocopy)
        {
            List<T>::resize_copy(currLen, capacity_);
        }

        capacity_ = List<T>::size();
        List<T>::setAddressableSize(currLen);
    }
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::resize
(
    const label len
)
{
    // Preserve all content
    resize_copy(List<T>::size(), len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::resize_copy
(
    label count,
    const label len
)
{
    // Preserve count elements
    doReserve_copy(count, len);
    List<T>::setAddressableSize(len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::resize_fill
(
    const label len,
    const T& val
)
{
    resize_nocopy(len);
    UList<T>::operator=(val);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::resize_nocopy
(
    const label len
)
{
    // Preserve 0 content
    resize_copy(0, len);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::resize
(
    const label len,
    const T& val
)
{
    const label oldLen = List<T>::size();

    // Preserve all content
    resize(len);

    // Fill newly exposed with constant value
    if (oldLen < List<T>::size())
    {
        std::fill
        (
            this->begin(oldLen), this->end(), val
        );
    }
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::clear() noexcept
{
    List<T>::setAddressableSize(0);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::clearStorage()
{
    // Extra safety...?
    if (!List<T>::cdata() && List<T>::empty()) capacity_ = 0;

    // Consistent allocated sizing
    List<T>::setAddressableSize(capacity_);
    List<T>::clear();
    capacity_ = 0;
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::shrink_to_fit()
{
    const label currLen = List<T>::size();
    if (currLen < capacity_)
    {
        List<T>::setAddressableSize(capacity_);
        List<T>::resize(currLen);
        capacity_ = List<T>::size();
    }
}


template<class T, int SizeMin>
inline void
Foam::DynamicField<T, SizeMin>::swap(List<T>& other)
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const List<T>*>(this)
         == static_cast<const List<T>*>(&other)
        )
    )
    {
        return;  // Self-swap is a no-op
    }

    // Remove unused storage
    this->shrink_to_fit();

    // Swap storage and addressable size
    UList<T>::swap(other);

    // Update capacity
    capacity_ = List<T>::size();
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline void Foam::DynamicField<T, SizeMin>::swap
(
    DynamicField<T, AnySizeMin>& other
) noexcept
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const List<T>*>(this)
         == static_cast<const List<T>*>(&other)
        )
    )
    {
        return;  // Self-swap is a no-op
    }

    // Swap storage and addressable size
    UList<T>::swap(other);

    // Swap capacity
    std::swap(this->capacity_, other.capacity_);
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline void Foam::DynamicField<T, SizeMin>::swap
(
    DynamicList<T, AnySizeMin>& other
) noexcept
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const List<T>*>(this)
         == static_cast<const List<T>*>(&other)
        )
    )
    {
        return;  // Self-swap is a no-op
    }

    // Swap storage and addressable size
    UList<T>::swap(other);

    // Swap capacity
    auto old = capacity_;
    this->setCapacity_unsafe(other.capacity());
    other.setCapacity_unsafe(old);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::transfer(List<T>& list)
{
    // No check for self-assignment (different types)

    // Consistent allocated sizing
    List<T>::setAddressableSize(capacity_);
    Field<T>::transfer(list);
    capacity_ = Field<T>::size();
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline void Foam::DynamicField<T, SizeMin>::transfer
(
    DynamicList<T, AnySizeMin>& other
)
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const UList<T>*>(this)
         == static_cast<const UList<T>*>(&other)
        )
    )
    {
        return;  // Self-assignment is a no-op
    }

    // Remove storage
    this->clearStorage();

    // Swap storage and addressable size
    UList<T>::swap(other);

    // Update capacity
    capacity_ = other.capacity();
    other.setCapacity_unsafe(0);
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline void Foam::DynamicField<T, SizeMin>::transfer
(
    DynamicField<T, AnySizeMin>& other
)
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const UList<T>*>(this)
         == static_cast<const UList<T>*>(&other)
        )
    )
    {
        return;  // Self-assignment is a no-op
    }

    // Remove storage
    this->clearStorage();

    // Swap storage and addressable size
    UList<T>::swap(other);

    // Update capacity
    capacity_ = other.capacity();
    other.setCapacity_unsafe(0);
}


template<class T, int SizeMin>
template<class... Args>
inline T& Foam::DynamicField<T, SizeMin>::emplace_back(Args&&... args)
{
    // This could/should be better with inplace construction
    // (as per std::vector), but currently lacking the methods for that
    // so resize and move assign

    const label idx = List<T>::size();
    resize(idx + 1);

    // move assign element
    this->operator[](idx) = T(std::forward<Args>(args)...);
    return this->operator[](idx);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::push_back
(
    const T& val
)
{
    const label idx = List<T>::size();
    resize(idx + 1);

    this->operator[](idx) = val;  // copy element
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::push_back
(
    T&& val
)
{
    const label idx = List<T>::size();
    resize(idx + 1);

    this->operator[](idx) = std::move(val);  // move assign element
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::push_back
(
    const UList<T>& list
)
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const UList<T>*>(this)
         == static_cast<const UList<T>*>(&list)
        )
    )
    {
        FatalErrorInFunction
            << "Attempted push_back to self"
            << abort(FatalError);
    }

    const label idx = List<T>::size();
    resize(idx + list.size());

    std::copy(list.begin(), list.end(), this->begin(idx));
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::push_back
(
    List<T>&& list
)
{
    if
    (
        FOAM_UNLIKELY
        (
            static_cast<const UList<T>*>(this)
         == static_cast<const UList<T>*>(&list)
        )
    )
    {
        FatalErrorInFunction
            << "Attempted push_back to self"
            << abort(FatalError);
    }

    const label idx = List<T>::size();
    resize(idx + list.size());

    // Move the elements
    std::move(list.begin(), list.end(), this->begin(idx));

    list.clear();
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::pop_back(label n)
{
    if (n >= this->size())
    {
        this->clear();
    }
    else if (n > 0)
    {
        resize(this->size() - n);
    }
}


// * * * * * * * * * * * * * * * Member Operators  * * * * * * * * * * * * * //

template<class T, int SizeMin>
inline T& Foam::DynamicField<T, SizeMin>::operator()
(
    const label i
)
{
    if (i >= List<T>::size())
    {
        resize(i + 1);
    }

    return this->operator[](i);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    const T& val
)
{
    UList<T>::operator=(val);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    Foam::zero
)
{
    UList<T>::operator=(Foam::zero{});
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    const UList<T>& list
)
{
    doAssignDynList(list);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    const DynamicField<T, SizeMin>& list
)
{
    if (this == &list)
    {
        return;  // Self-assignment is a no-op
    }

    doAssignDynList(list);
}


template<class T, int SizeMin>
template<class Addr>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    const IndirectListBase<T, Addr>& list
)
{
    // NOTE: Self-assignment needs special handling
    /// if
    /// (
    ///     static_cast<const UList<T>*>(this)
    ///  == static_cast<const UList<T>*>(&list.values())
    /// )

    doAssignDynList(list);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    List<T>&& list
)
{
    this->transfer(list);
}


template<class T, int SizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    DynamicField<T, SizeMin>&& list
)
{
    this->transfer(list);
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    DynamicField<T, AnySizeMin>&& list
)
{
    this->transfer(list);
}


template<class T, int SizeMin>
template<int AnySizeMin>
inline void Foam::DynamicField<T, SizeMin>::operator=
(
    DynamicList<T, AnySizeMin>&& list
)
{
    this->transfer(list);
}


// * * * * * * * * * * * * * * * IOstream Operators  * * * * * * * * * * * * //

template<class T, int SizeMin>
inline Foam::Istream& Foam::DynamicField<T, SizeMin>::readList
(
    Istream& is
)
{
    // Use DynamicList::readList for reading DynamicField.
    // The logic should be the same and this avoids duplicate code

    DynamicList<T, SizeMin> list;
    this->swap(list);

    list.readList(is);
    this->swap(list);

    return is;
}


template<class T, int SizeMin>
inline Foam::Istream& Foam::operator>>
(
    Istream& is,
    DynamicField<T, SizeMin>& rhs
)
{
    return rhs.readList(is);
}


template<class T, int SizeMin>
inline Foam::Ostream& Foam::operator<<
(
    Ostream& os,
    const DynamicField<T, SizeMin>& rhs
)
{
    os << static_cast<const Field<T>&>(rhs);
    return os;
}


// ************************************************************************* //
