/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.model.database;

import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import org.jabref.architecture.AllowedToUseLogic;
import org.jabref.gui.LibraryTab;
import org.jabref.gui.desktop.JabRefDesktop;
import org.jabref.logic.shared.DatabaseLocation;
import org.jabref.logic.shared.DatabaseSynchronizer;
import org.jabref.logic.util.CoarseChangeFilter;
import org.jabref.logic.util.io.BackupFileUtil;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseMode;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.metadata.MetaData;
import org.jabref.preferences.FilePreferences;
import org.jspecify.annotations.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@AllowedToUseLogic(value="because it needs access to shared database features")
public class BibDatabaseContext {
    private static final Logger LOGGER = LoggerFactory.getLogger(LibraryTab.class);
    private final BibDatabase database;
    private MetaData metaData;
    private final String uid = "bibdatabasecontext_" + String.valueOf(UUID.randomUUID());
    private Path path;
    private DatabaseSynchronizer dbmsSynchronizer;
    private CoarseChangeFilter dbmsListener;
    private DatabaseLocation location;

    public BibDatabaseContext() {
        this(new BibDatabase());
    }

    public BibDatabaseContext(BibDatabase database) {
        this(database, new MetaData());
    }

    public BibDatabaseContext(BibDatabase database, MetaData metaData) {
        this.database = Objects.requireNonNull(database);
        this.metaData = Objects.requireNonNull(metaData);
        this.location = DatabaseLocation.LOCAL;
    }

    public BibDatabaseContext(BibDatabase database, MetaData metaData, Path path) {
        this(database, metaData, path, DatabaseLocation.LOCAL);
    }

    public BibDatabaseContext(BibDatabase database, MetaData metaData, Path path, DatabaseLocation location) {
        this(database, metaData);
        Objects.requireNonNull(location);
        this.path = path;
        if (location == DatabaseLocation.LOCAL) {
            this.convertToLocalDatabase();
        }
    }

    public BibDatabaseMode getMode() {
        return this.metaData.getMode().orElse(BibDatabaseMode.BIBLATEX);
    }

    public void setMode(BibDatabaseMode bibDatabaseMode) {
        this.metaData.setMode(bibDatabaseMode);
    }

    public void setDatabasePath(Path file) {
        this.path = file;
    }

    public Optional<Path> getDatabasePath() {
        return Optional.ofNullable(this.path);
    }

    public void clearDatabasePath() {
        this.path = null;
    }

    public BibDatabase getDatabase() {
        return this.database;
    }

    public MetaData getMetaData() {
        return this.metaData;
    }

    public void setMetaData(MetaData metaData) {
        this.metaData = Objects.requireNonNull(metaData);
    }

    public boolean isBiblatexMode() {
        return this.getMode() == BibDatabaseMode.BIBLATEX;
    }

    public boolean isStudy() {
        return this.getDatabasePath().map(path -> path.getFileName().toString().equals("studyResult.bib") && Files.exists(path.resolveSibling("study.yml"), new LinkOption[0])).orElse(false);
    }

    public List<Path> getFileDirectories(FilePreferences preferences) {
        ArrayList fileDirs = new ArrayList();
        this.metaData.getUserFileDirectory(preferences.getUserAndHost()).ifPresent(userFileDirectory -> fileDirs.add(this.getFileDirectoryPath((String)userFileDirectory)));
        this.metaData.getDefaultFileDirectory().ifPresent(metaDataDirectory -> fileDirs.add(this.getFileDirectoryPath((String)metaDataDirectory)));
        if (fileDirs.isEmpty() && preferences.shouldStoreFilesRelativeToBibFile()) {
            this.getDatabasePath().ifPresent(dbPath -> {
                Path parentPath = dbPath.getParent();
                if (parentPath == null) {
                    parentPath = Path.of(System.getProperty("user.dir"), new String[0]);
                }
                Objects.requireNonNull(parentPath, "BibTeX database parent path is null");
                fileDirs.add(parentPath);
            });
        } else {
            preferences.getMainFileDirectory().ifPresent(fileDirs::add);
        }
        return fileDirs.stream().map(Path::toAbsolutePath).collect(Collectors.toList());
    }

    public Optional<Path> getFirstExistingFileDir(FilePreferences preferences) {
        return this.getFileDirectories(preferences).stream().filter(x$0 -> Files.exists(x$0, new LinkOption[0])).findFirst();
    }

    private Path getFileDirectoryPath(String directoryName) {
        Path directory = Path.of(directoryName, new String[0]);
        Optional<Path> databaseFile = this.getDatabasePath();
        if (!directory.isAbsolute() && databaseFile.isPresent()) {
            return databaseFile.get().getParent().resolve(directory).normalize();
        }
        return directory;
    }

    public DatabaseSynchronizer getDBMSSynchronizer() {
        return this.dbmsSynchronizer;
    }

    public void clearDBMSSynchronizer() {
        this.dbmsSynchronizer = null;
    }

    public DatabaseLocation getLocation() {
        return this.location;
    }

    public void convertToSharedDatabase(DatabaseSynchronizer dmbsSynchronizer) {
        this.dbmsSynchronizer = dmbsSynchronizer;
        this.dbmsListener = new CoarseChangeFilter(this);
        this.dbmsListener.registerListener(this.dbmsSynchronizer);
        this.location = DatabaseLocation.SHARED;
    }

    public void convertToLocalDatabase() {
        if (this.dbmsListener != null && this.location == DatabaseLocation.SHARED) {
            this.dbmsListener.unregisterListener(this.dbmsSynchronizer);
            this.dbmsListener.shutdown();
        }
        this.location = DatabaseLocation.LOCAL;
    }

    public List<BibEntry> getEntries() {
        return this.database.getEntries();
    }

    public @NonNull Path getFulltextIndexPath() {
        Path appData = JabRefDesktop.getFulltextIndexBaseDirectory();
        if (this.getDatabasePath().isPresent()) {
            Path databasePath = this.getDatabasePath().get();
            String fileName = BackupFileUtil.getUniqueFilePrefix(databasePath) + "--" + String.valueOf(databasePath.getFileName());
            Path indexPath = appData.resolve(fileName);
            LOGGER.debug("Index path for {} is {}", (Object)this.getDatabasePath().get(), (Object)indexPath);
            return indexPath;
        }
        Path indexPath = appData.resolve("unsaved");
        LOGGER.debug("Using index for unsaved database: {}", (Object)indexPath);
        return indexPath;
    }

    public String toString() {
        return "BibDatabaseContext{metaData=" + String.valueOf(this.metaData) + ", mode=" + String.valueOf((Object)this.getMode()) + ", databasePath=" + String.valueOf(this.getDatabasePath()) + ", biblatexMode=" + this.isBiblatexMode() + ", uid= " + this.getUid() + ", fulltextIndexPath=" + String.valueOf(this.getFulltextIndexPath()) + "}";
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof BibDatabaseContext)) {
            return false;
        }
        BibDatabaseContext that = (BibDatabaseContext)o;
        return Objects.equals(this.database, that.database) && Objects.equals(this.metaData, that.metaData) && Objects.equals(this.path, that.path) && this.location == that.location;
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.database, this.metaData, this.path, this.location});
    }

    public String getUid() {
        return this.uid;
    }
}

