import type { ChatCompletionAssistantMessageParam } from 'openai/resources/chat/completions';
import { ChatCompletionChunk } from 'openai/resources';
import type { ChatCompletionMessage } from 'openai/resources/chat/completions';
import type { ChatCompletionMessageParam } from 'openai/resources/chat/completions';
import { ChatCompletionMessageToolCall } from 'openai/resources';
import type { ChatCompletionTool } from 'openai/resources/chat/completions';
import type { ChatCompletionToolChoiceOption } from 'openai/resources/chat/completions';
import { ChatCompletionToolMessageParam } from 'openai/resources/chat/completions';
import { ChatCompletionUserMessageParam } from 'openai/resources/chat/completions';
import { Client } from '@modelcontextprotocol/sdk/client/index.js';
import { JSONRPCMessage } from '@modelcontextprotocol/sdk/types.js';
import { McpServer } from '@modelcontextprotocol/sdk/server/mcp.js';
import { OAuthClientProvider } from '@modelcontextprotocol/sdk/client/auth.js';
import { OpenAI } from 'openai/index.mjs';
import type { ReasoningEffort } from 'openai/resources';
import { SSEClientTransport } from '@modelcontextprotocol/sdk/client/sse.js';
import { StdioClientTransport } from '@modelcontextprotocol/sdk/client/stdio.js';
import { StreamableHTTPClientTransport } from '@modelcontextprotocol/sdk/client/streamableHttp.js';
import { Transport } from '@modelcontextprotocol/sdk/shared/transport.js';
import { Writable } from 'stream';
import * as z from 'zod';
import { z as z_2 } from 'zod';

export declare type AbortEvent = z.infer<typeof AbortEventSchema>;

/**
 * Turn abort event - User aborted the current turn
 * Triggers completion of orphaned tool calls
 */
declare const AbortEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"abort">;
    data: z.ZodObject<{
        reason: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        reason: string;
    }, {
        reason: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        reason: string;
    };
    id: string;
    type: "abort";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        reason: string;
    };
    id: string;
    type: "abort";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

declare type AgentAction = (typeof AgentActions)[number];

declare const AgentActions: readonly ["fix", "fix-pr-comment", "task"];

declare type AgentIdBag = {
    /**
     * The ID of the agent whose invocation generated the object this bag is attached to.
     */
    agentId?: string;
};

declare type ApiKeyAuthInfo = {
    readonly type: "api-key";
    readonly apiKey: string;
    readonly host: string;
};

/**
 * An assessed command, with its identifier and whether it is read-only.
 */
declare type AssessedCommand = {
    /**
     * The command identifier, e.g. "rm" or "git push".
     */
    readonly identifier: string;
    /**
     * Whether the command is read-only (i.e. does not modify state).
     */
    readOnly: boolean;
};

export declare type AssistantIntentEvent = z.infer<typeof AssistantIntentEventSchema>;

declare const AssistantIntentEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.intent">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        intent: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        intent: string;
    }, {
        intent: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        intent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.intent";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        intent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.intent";
    timestamp: string;
    parentId: string | null;
}>;

export declare type AssistantMessageDeltaEvent = z.infer<typeof AssistantMessageDeltaEventSchema>;

/**
 * Assistant message streaming delta (ephemeral, not persisted)
 * Sent when streaming is enabled to provide incremental response chunks.
 * Clients should accumulate deltaContent to build the full response.
 */
declare const AssistantMessageDeltaEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.message_delta">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        messageId: z.ZodString;
        deltaContent: z.ZodString;
        totalResponseSizeBytes: z.ZodOptional<z.ZodNumber>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    }, {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.message_delta";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.message_delta";
    timestamp: string;
    parentId: string | null;
}>;

export declare type AssistantMessageEvent = z.infer<typeof AssistantMessageEventSchema>;

/**
 * An event that is emitted by the `Client` for each message it receives from the LLM.
 *
 * Currently does not include telemetry.
 */
declare type AssistantMessageEvent_2 = {
    kind: "message";
    turn?: number;
    callId?: string;
    modelCall?: ModelCallParam;
    message: ChatCompletionMessageParamsWithToolCalls & ReasoningMessageParam;
};

/**
 * Assistant text response (with optional tool requests)
 */
declare const AssistantMessageEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.message">;
    data: z.ZodObject<{
        messageId: z.ZodString;
        content: z.ZodString;
        toolRequests: z.ZodOptional<z.ZodArray<z.ZodObject<{
            toolCallId: z.ZodString;
            name: z.ZodString;
            arguments: z.ZodUnknown;
            type: z.ZodOptional<z.ZodEnum<["function", "custom"]>>;
        }, "strip", z.ZodTypeAny, {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }, {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }>, "many">>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    }, {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "assistant.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "assistant.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type AssistantReasoningDeltaEvent = z.infer<typeof AssistantReasoningDeltaEventSchema>;

/**
 * Assistant reasoning streaming delta (ephemeral, not persisted)
 * Sent when streaming is enabled to provide incremental reasoning chunks.
 * Clients should accumulate deltaContent to build the full reasoning.
 */
declare const AssistantReasoningDeltaEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.reasoning_delta">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        reasoningId: z.ZodString;
        deltaContent: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        reasoningId: string;
        deltaContent: string;
    }, {
        reasoningId: string;
        deltaContent: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        reasoningId: string;
        deltaContent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.reasoning_delta";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        reasoningId: string;
        deltaContent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.reasoning_delta";
    timestamp: string;
    parentId: string | null;
}>;

export declare type AssistantReasoningEvent = z.infer<typeof AssistantReasoningEventSchema>;

/**
 * Assistant reasoning content (chain-of-thought), persisted as final content
 */
declare const AssistantReasoningEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.reasoning">;
    data: z.ZodObject<{
        reasoningId: z.ZodString;
        content: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        content: string;
        reasoningId: string;
    }, {
        content: string;
        reasoningId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        content: string;
        reasoningId: string;
    };
    id: string;
    type: "assistant.reasoning";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        content: string;
        reasoningId: string;
    };
    id: string;
    type: "assistant.reasoning";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type AssistantTurnEndEvent = z.infer<typeof AssistantTurnEndEventSchema>;

/**
 * Agent completes a turn
 */
declare const AssistantTurnEndEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.turn_end">;
    data: z.ZodObject<{
        turnId: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        turnId: string;
    }, {
        turnId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type AssistantTurnStartEvent = z.infer<typeof AssistantTurnStartEventSchema>;

/**
 * Agent starts processing a turn
 */
declare const AssistantTurnStartEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.turn_start">;
    data: z.ZodObject<{
        turnId: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        turnId: string;
    }, {
        turnId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type AssistantUsageEvent = z.infer<typeof AssistantUsageEventSchema>;

/**
 * Assistant usage metrics (ephemeral event for UI updates)
 * Used to track model usage, tokens, costs, and durations without persisting to disk
 * Emitted in app.tsx's onModelCallSuccess callback
 */
declare const AssistantUsageEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.usage">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        model: z.ZodOptional<z.ZodString>;
        inputTokens: z.ZodOptional<z.ZodNumber>;
        outputTokens: z.ZodOptional<z.ZodNumber>;
        cacheReadTokens: z.ZodOptional<z.ZodNumber>;
        cacheWriteTokens: z.ZodOptional<z.ZodNumber>;
        cost: z.ZodOptional<z.ZodNumber>;
        duration: z.ZodOptional<z.ZodNumber>;
        initiator: z.ZodOptional<z.ZodString>;
        apiCallId: z.ZodOptional<z.ZodString>;
        providerCallId: z.ZodOptional<z.ZodString>;
        quotaSnapshots: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodObject<{
            isUnlimitedEntitlement: z.ZodBoolean;
            entitlementRequests: z.ZodNumber;
            usedRequests: z.ZodNumber;
            usageAllowedWithExhaustedQuota: z.ZodBoolean;
            overage: z.ZodNumber;
            overageAllowedWithExhaustedQuota: z.ZodBoolean;
            remainingPercentage: z.ZodNumber;
            resetDate: z.ZodOptional<z.ZodDate>;
        }, "strip", z.ZodTypeAny, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }>>>;
    }, "strip", z.ZodTypeAny, {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    }, {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.usage";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.usage";
    timestamp: string;
    parentId: string | null;
}>;

export declare type Attachment = z.infer<typeof AttachmentSchema>;

declare const AttachmentSchema: z.ZodDiscriminatedUnion<"type", [z.ZodObject<{
    path: z.ZodString;
    displayName: z.ZodString;
} & {
    type: z.ZodLiteral<"file">;
}, "strip", z.ZodTypeAny, {
    type: "file";
    path: string;
    displayName: string;
}, {
    type: "file";
    path: string;
    displayName: string;
}>, z.ZodObject<{
    path: z.ZodString;
    displayName: z.ZodString;
} & {
    type: z.ZodLiteral<"directory">;
}, "strip", z.ZodTypeAny, {
    type: "directory";
    path: string;
    displayName: string;
}, {
    type: "directory";
    path: string;
    displayName: string;
}>, z.ZodObject<{
    type: z.ZodLiteral<"selection">;
    filePath: z.ZodString;
    displayName: z.ZodString;
    text: z.ZodString;
    selection: z.ZodObject<{
        start: z.ZodObject<{
            line: z.ZodNumber;
            character: z.ZodNumber;
        }, "strip", z.ZodTypeAny, {
            line: number;
            character: number;
        }, {
            line: number;
            character: number;
        }>;
        end: z.ZodObject<{
            line: z.ZodNumber;
            character: z.ZodNumber;
        }, "strip", z.ZodTypeAny, {
            line: number;
            character: number;
        }, {
            line: number;
            character: number;
        }>;
    }, "strip", z.ZodTypeAny, {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    }, {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    }>;
}, "strip", z.ZodTypeAny, {
    type: "selection";
    text: string;
    filePath: string;
    displayName: string;
    selection: {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    };
}, {
    type: "selection";
    text: string;
    filePath: string;
    displayName: string;
    selection: {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    };
}>]>;

/**
 * Represents the authentication information for a user.
 */
declare type AuthInfo = HMACAuthInfo | EnvAuthInfo | UserAuthInfo | GhCliAuthInfo | ApiKeyAuthInfo | TokenAuthInfo | CopilotApiTokenAuthInfo;

/**
 * Base interface for all hook inputs
 */
export declare interface BaseHookInput {
    timestamp: number;
    cwd: string;
}

export declare abstract class BaseLogger implements RunnerLogger {
    protected logLevel?: LogLevel;
    protected debugEnvironmentVariables?: string[];
    private secretFilter;
    constructor(logLevel?: LogLevel, debugEnvironmentVariables?: string[]);
    filterSecrets(messageOrError: string | Error): string | Error;
    /**
     * Returns true if the log level is not set, or the log level is set and the level is enabled.
     */
    shouldLog(level: LogLevel): boolean;
    isDebug(): boolean;
    abstract log(message: string): void;
    abstract info(message: string): void;
    abstract debug(message: string): void;
    abstract notice(message: string | Error): void;
    abstract warning(message: string | Error): void;
    abstract error(message: string | Error): void;
    abstract startGroup(name: string, level?: LogLevel): void;
    abstract endGroup(level?: LogLevel): void;
}

declare type BasicToolConfig = {
    serverName: string;
    name: string;
    namespacedName: string;
    mcpServerName?: string;
    mcpToolName?: string;
    title: string;
    description: string;
    input_schema: ToolInputSchema;
    readOnly?: boolean;
    safeForTelemetry?: Tool_2["safeForTelemetry"] & Tool["safeForTelemetry"];
    filterMode?: ContentFilterMode;
};

declare type BinaryResult = {
    data: string;
    mimeType: string;
    type: string;
    /**
     * A description of the binary data.
     */
    description?: string;
};

declare type ChatCompletionMessageParamsWithToolCalls = Omit<ChatCompletionAssistantMessageParam, "tool_calls"> & {
    tool_calls?: CopilotChatCompletionMessageToolCall[];
    copilot_annotations?: unknown;
};

/**
 * Information about a checkpoint for display in the prompt.
 */
declare interface CheckpointInfo {
    /** Checkpoint number (1-indexed) */
    number: number;
    /** Title of the checkpoint */
    title: string;
    /** Filename of the checkpoint (e.g., "001-plan-design.md") */
    filename: string;
}

declare interface Client_2 {
    readonly model: string;
    getCompletionWithTools(systemMessage: string, initialMessages: ChatCompletionMessageParam[], tools: Tool_2[], options?: GetCompletionWithToolsOptions): AsyncGenerator<Event_2>;
}

declare interface ClientFactory {
    createClient(options: any, capabilities: any): Client;
}

/**
 * Connection information for an MCP client.
 */
declare type ClientInfo = {
    /**
     * Name of the MCP server this connection is for.
     */
    clientName: string;
    /**
     * Optional original/display name for this server.
     * This may contain "/" when the internal `clientName` has been adapted (e.g. "/" -> "__").
     */
    displayName?: string;
    /**
     * MCP client instance connected to the server.
     */
    mcpClient: Client;
    /**
     * Telemetry configuration for this connection.
     */
    safeForTelemetry?: Tool_2["safeForTelemetry"];
    /**
     * List of tools from the server to expose. ["*"] means all tools.
     */
    tools: string[];
    /**
     * Filter mode for the tools from this client.
     * If not specified, defaults to ContentFilterMode.HiddenCharacters.
     * If specified as a map, it applies to each tool by name.
     */
    filterMapping?: Record<string, ContentFilterMode> | ContentFilterMode;
    /**
     * Optional timeout in milliseconds for tool calls to this server.
     * If not specified, uses the default timeout.
     */
    timeout?: number;
};

/**
 * The ideal set of options that a `{@link Client}` expose.
 */
declare type ClientOptions = {
    /**
     * The model to use for LLM completions.
     */
    model?: string;
    /**
     * The proportion of the model's input/prompt token limit
     * that should be given to tools as their token budget.
     */
    toolTokenBudgetProportion?: number;
    retryPolicy?: ClientRetryPolicy;
    /**
     * If for the current model, a higher level of thinking is possible, use it.
     * @default false
     */
    thinkingMode?: boolean;
    /**
     * The token budget for extended thinking/chain-of-thought for models that support it.
     * For Anthropic Claude models via CAPI, this maps to the `thinking_budget` parameter.
     * When set, enables extended thinking with the specified token budget.
     * This field remains optional even in ClientOptionsRequired since not all models support it.
     */
    thinkingBudget?: number | undefined;
    requestHeaders?: Record<string, string>;
    /**
     * If true, enables cache control checkpoints on messages sent to the model.
     * This allows downstream services to better manage caching of responses.
     * Defaults to false.
     */
    enableCacheControl?: boolean;
    /**
     * The default reasoning effort level for the model to use, if supported by the client.
     */
    defaultReasoningEffort?: ReasoningEffort;
};

/**
 * Retry policies for the AI client.
 */
declare type ClientRetryPolicy = {
    /**
     * The maximum number of retries for **any** type of retryable failure or error.
     */
    maxRetries?: number;
    /**
     * Specific error codes that should always be retried.
     * - If a `number`, that specific error code will be retried.
     * - If a `[number, number]`, all error codes in the range will be retried (inclusive).
     * - If a `[number, undefined]`, all error codes greater than or equal to the first number will be retried.
     * - To retry all error codes based on an upper bound, simply use `[0, number]`.
     *
     * Some error codes are retried by default even if not specified here, for example 429 (rate limit exceeded).
     */
    errorCodesToRetry?: (number | [number, number | undefined])[];
    /**
     * How to handle retries for rate limiting (429) errors. If a policy is not provided, a default
     * policy will be used.
     */
    rateLimitRetryPolicy?: {
        /**
         * The default wait time in between retries if the server does not
         * provide a `retry-after` header.
         */
        defaultRetryAfterSeconds?: number;
        /**
         * The initial extra wait time in between retries. The extra wait time will
         * be added to the `retry-after` header value (or {@link defaultRetryAfterSeconds} if
         * the header is not present). After each retry, the extra wait time will grow beyond
         * this value according to the {@link retryBackoffExtraGrowth} factor.
         */
        initialRetryBackoffExtraSeconds?: number;
        /**
         * The growth factor for the retry backoff extra time. E.g. 2x, 3x, etc.
         */
        retryBackoffExtraGrowth?: number;
        /**
         * The maximum wait time in between retries.
         */
        maxRetryAfterSeconds?: number;
    };
};

declare type Command = {
    readonly identifier: string;
    readonly readOnly: boolean;
};

declare type CompactionCompletedEvent = {
    kind: "compaction_completed";
    turn: number;
    performedBy: string;
    success: boolean;
    error?: string;
    compactionResult?: CompactionEventResult;
};

declare type CompactionEvent = CompactionStartedEvent | CompactionCompletedEvent;

declare type CompactionEventResult = {
    tokenLimit: number;
    preCompactionTokens: number;
    preCompactionMessagesLength: number;
    postCompactionTokens?: number;
    postCompactionMessagesLength?: number;
    tokensRemoved?: number;
    messagesRemoved?: number;
    summaryContent: string;
    compactionTokensUsed?: {
        input: number;
        output: number;
        cachedInput: number;
    };
};

/**
 * Result of a conversation history compaction operation.
 */
export declare interface CompactionResult {
    success: boolean;
    tokensRemoved: number;
    messagesRemoved: number;
    summaryContent: string;
}

declare type CompactionStartedEvent = {
    kind: "compaction_started";
    turn: number;
    performedBy: string;
};

export declare function completeOrphanedToolCalls(messages: ChatCompletionMessageParam[]): ChatCompletionMessageParam[];

declare type CompletionWithToolsModel = {
    readonly name: string;
    readonly id?: string;
    readonly capabilities?: {
        readonly supports?: {
            readonly vision?: boolean;
        };
        readonly limits?: {
            readonly max_prompt_tokens?: number;
            readonly max_context_window_tokens?: number;
            readonly vision?: {
                readonly supported_media_types: string[];
                readonly max_prompt_images: number;
                readonly max_prompt_image_size: number;
            };
        };
    };
};

export declare class CompoundLogger implements RunnerLogger {
    readonly loggers: RunnerLogger[];
    constructor(loggers: RunnerLogger[]);
    isDebug(): boolean;
    debug(message: string): void;
    log(message: string): void;
    info(message: string): void;
    notice(message: string | Error): void;
    warning(message: string | Error): void;
    error(message: string | Error): void;
    startGroup(name: string, level?: LogLevel): void;
    endGroup(level?: LogLevel): void;
}

export declare class ConsoleLogger extends BaseLogger implements RunnerLogger {
    constructor(logLevel?: LogLevel, debugEnvironmentVariables?: string[]);
    log(message: string): void;
    debug(message: string): void;
    info(message: string): void;
    notice(message: string | Error): void;
    warning(message: string | Error): void;
    error(message: string | Error): void;
    startGroup(name: string, level?: LogLevel): void;
    endGroup(level?: LogLevel): void;
}

declare enum ContentFilterMode {
    None = "none",
    Markdown = "markdown",
    HiddenCharacters = "hidden_characters"
}

/** Represents direct Copilot API authentication (via GITHUB_COPILOT_API_TOKEN + COPILOT_API_URL). */
declare type CopilotApiTokenAuthInfo = {
    readonly type: "copilot-api-token";
    readonly host: "https://github.com";
};

/**
 * Note: agent sessions API depend on this type!
 */
declare type CopilotChatCompletionChunk = Omit<ChatCompletionChunk, "choices"> & {
    choices: CopilotChatCompletionChunkChoices;
};

declare type CopilotChatCompletionChunkChoice = Omit<ChatCompletionChunk.Choice, "delta"> & {
    delta: CopilotChatCompletionChunkDelta;
};

declare type CopilotChatCompletionChunkChoices = Array<CopilotChatCompletionChunkChoice>;

declare type CopilotChatCompletionChunkDelta = Omit<ChatCompletionChunk.Choice.Delta, "tool_calls"> & ReasoningMessageParam & {
    tool_calls?: Array<CopilotChatCompletionToolCallDelta>;
    copilot_annotations?: string | undefined;
};

/**
 * Re-export the OpenAI union type for convenience.
 * ChatCompletionMessageToolCall = ChatCompletionMessageFunctionToolCall | ChatCompletionMessageCustomToolCall
 */
declare type CopilotChatCompletionMessageToolCall = ChatCompletionMessageToolCall & {
    index?: number;
};

/**
 * Streaming tool call delta that supports both function and custom tool calls.
 */
declare type CopilotChatCompletionToolCallDelta = FunctionToolCallDelta | CustomToolCallDelta;

declare interface CopilotSessionsClient {
    sessionId(): string;
    error(error: Error): Promise<void>;
    log(logs: SessionLogsContent): Promise<void>;
    logNonCompletionContent(content: string): Promise<void>;
    createOrUpdateMCPStartupToolCall(params: {
        content?: string;
        serverName: string;
        toolNamesToDisplayNames?: Record<string, string>;
    }): Promise<void>;
    logTitleAndBody(title: string, body: string, agentId?: string): Promise<void>;
    getLogs(sessionId: string): Promise<SessionLogsContents | string | undefined>;
}

/**
 * Type for a custom tool call delta (streaming).
 */
declare type CustomToolCallDelta = {
    index: number;
    id?: string;
    type?: "custom";
    custom?: {
        name?: string;
        input?: string;
    };
};

/**
 * Format specification for custom tools that use grammar-based input.
 */
declare type CustomToolInputFormat = {
    /**
     * The type of format. Currently only "grammar" is supported.
     */
    type: "grammar";
    /**
     * The syntax of the grammar (e.g., "lark").
     */
    syntax: string;
    /**
     * The grammar definition.
     */
    definition: string;
};

declare type DeepPartial<T> = {
    [P in keyof T]?: T[P] extends (infer U)[] ? U[] : T[P] extends object ? DeepPartial<T[P]> : T[P];
};

export declare const DEFAULT_INTEGRATION_ID = "copilot-developer-cli";

declare const defaultFeatureFlags: {
    CUSTOM_AGENTS: boolean;
    CCA_DELEGATE: boolean;
    CONTINUITY: boolean;
    CONTINUITY_REMOTE_CONTROL: boolean;
    FEATURE_FLAG_TEST: boolean;
    S2STOKENS: boolean;
    "copilot-feature-agentic-memory": boolean;
    PLAN_COMMAND: boolean;
    PLUGIN_COMMAND: boolean;
    COPILOT_SWE_AGENT_UNIFIED_TASK_TOOL: boolean;
    COPILOT_SWE_AGENT_BACKGROUND_AGENTS: boolean;
    COPILOT_SWE_AGENT_EXPANDED_BUILTIN_AGENTS: boolean;
    COPILOT_SWE_AGENT_PARALLEL_TASK_EXECUTION: boolean;
    VSCODE_INTEGRATION: boolean;
    DIFF_MODE: boolean;
    UNDO_REWIND: boolean;
    QUEUED_COMMANDS: boolean;
};

/**
 * Enables a model's policy for the current user.
 * This is used for models that are available but require user consent (policy.state !== 'enabled').
 */
export declare function enableModelPolicy(authInfo: AuthInfo, modelId: string, integrationId?: string, sessionId?: string, logger?: RunnerLogger): Promise<EnableModelPolicyResult>;

export declare type EnableModelPolicyResult = {
    success: boolean;
    error?: string;
    canBeEnabled?: boolean;
};

/** Represents the Personal Access Token (PAT) or server-to-server token authentication information. */
declare type EnvAuthInfo = {
    readonly type: "env";
    readonly host: string;
    /** The login of the user. Undefined for server-to-server tokens (ghs_). */
    readonly login?: string;
    readonly token: string;
    readonly envVar: string;
};

/**
 * EnvValueMode controls how the value of a key/value pair in the `env` block of an MCP server config is
 * interpreted. `direct` means the value is used as-is. `indirect` means the value is the name
 * of an environment variable whose value should be used.
 */
declare type EnvValueMode = "direct" | "indirect";

export declare type ErrorOccurredHook = (input: ErrorOccurredHookInput) => Promise<ErrorOccurredHookOutput | void>;

/**
 * Error occurred hook types
 */
export declare interface ErrorOccurredHookInput extends BaseHookInput {
    error: Error;
    errorContext: "model_call" | "tool_execution" | "system" | "user_input";
    recoverable: boolean;
}

export declare interface ErrorOccurredHookOutput {
    suppressOutput?: boolean;
    errorHandling?: "retry" | "skip" | "abort";
    retryCount?: number;
    userNotification?: string;
}

declare type Event_2 = MessageEvent_2 | ResponseEvent | ModelCallFailureEvent | TurnEvent | TruncationEvent | UsageInfoEvent | CompactionEvent | ImageProcessingEvent | ImageRemovalEvent | ModelCallSuccessEvent | ToolExecutionEvent;

declare type EventData<T extends EventType> = EventPayload<T>["data"];

declare type EventHandler<T extends EventType> = (event: EventPayload<T>) => void;

declare type EventPayload<T extends EventType> = Extract<SessionEvent, {
    type: T;
}>;

/**
 * Telemetry can be emitted by the runtime via progress events. Telemetry is attached to progress events
 * on a `telemetry` property whose type is this.
 */
declare type EventTelemetry<EventT = string, TelemetryT extends Telemetry = Telemetry> = {
    /**
     * The name of the telemetry event associated with the emitted runtime progress event.
     */
    event: EventT;
    /**
     * String-esque properties that are associated with the telemetry event.
     * WARNING: Do not put sensitive data here. Use restrictedProperties for that.
     */
    properties: TelemetryT["properties"];
    /**
     * String-esque properties that are associated with the telemetry event. These are only available on the restricted topics
     */
    restrictedProperties: TelemetryT["restrictedProperties"];
    /**
     * Number-esque metrics that are associated with the telemetry event. Both integer and floating point values are possible.
     */
    metrics: TelemetryT["metrics"];
};

declare type EventType = SessionEvent["type"];

export declare function executeHooks<TInput extends BaseHookInput, TOutput>(hooks: ((input: TInput) => Promise<TOutput | void>)[] | undefined, input: TInput, logger: RunnerLogger): Promise<void | TOutput>;

export declare interface ExternalToolDefinition {
    name: string;
    description: string;
    parameters?: Record<string, unknown>;
}

export declare type ExternalToolDispatcher = (request: ExternalToolInvocation) => Promise<ToolResult>;

export declare interface ExternalToolInvocation {
    sessionId: string;
    toolCallId: string;
    toolName: string;
    arguments: unknown;
    abortSignal?: AbortSignal;
}

declare type FeatureFlags = Readonly<typeof defaultFeatureFlags>;

export declare class FileLogger extends BaseLogger implements RunnerLogger, LogWriter {
    private readonly filePath;
    /** Promise that resolves when pending writes are complete. Used to serialize
     * writes and for testing. */
    writeQueue: Promise<void>;
    constructor(filePath: string, logLevel?: LogLevel, debugEnvironmentVariables?: string[]);
    outputPath(): string;
    log(message: string): void;
    debug(message: string): void;
    info(message: string): void;
    notice(message: string | Error): void;
    warning(message: string | Error): void;
    error(message: string | Error): void;
    startGroup(name: string, level?: LogLevel): void;
    endGroup(level?: LogLevel): void;
    write(category: string, message: string): void;
    /** Implementation for LogWriter interface */
    writeLog(level: LogLevel_2, message: string): Promise<void>;
    private performWrite;
}

/**
 * Type for a function tool call delta (streaming).
 */
declare type FunctionToolCallDelta = {
    index: number;
    id?: string;
    type?: "function";
    function?: {
        name?: string;
        arguments?: string;
    };
};

/**
 * Retrieves the list of available models based on policies and integration including capabilities and
 * billing information, which may be cached from previous calls.
 *
 * This list is in order of preference to be the default model for new sessions where the first model is
 * the most preferred.  It can be empty if all model policies are disabled.
 *
 * @deprecated Used solely by vscode-copilot-chat extension, use {@link retrieveAvailableModels} instead.
 */
export declare function getAvailableModels(authInfo: AuthInfo): Promise<Model[]>;

declare type GetCompletionWithToolsOptions = {
    /**
     * If `true`, then calls do `getCompletionWithTools` will check if the token counts of
     * the initial system messages, user messages, and tool definitions, exceed the limits
     * of the model. If they do, then the call will throw an error. If `false`, then the
     * call will not perform any checks.
     *
     * Defaults to `false`.
     */
    failIfInitialInputsTooLong?: boolean;
    toolChoice?: ChatCompletionToolChoiceOption;
    requestHeaders?: Record<string, string>;
    /**
     * If true, performs the request in streaming mode. This results in additional events
     * as each chunk is received from the service.
     */
    stream?: boolean;
    /**
     * If this call is a continuation of a previous `getCompletionWithTools` call, this specifies what turn
     * that conversation was/is on. This is used to determine the initial turn count in this
     * call to `getCompletionWithTools`.
     */
    initialTurnCount?: number;
    /**
     * If `true`, then the client will execute tools in parallel if model supports `parallel_tool_calls`.
     * If `false`, then the client will execute tools sequentially even if the model returns multiple tool calls.
     * Defaults to false.
     */
    executeToolsInParallel?: boolean;
    /**
     * Processors provide a way to do work during different stages of the completion with tools
     * lifecycle. Processors will be called in the order they are provided.
     */
    processors?: {
        preRequest?: IPreRequestProcessor[];
        onRequestError?: IOnRequestErrorProcessor[];
        preToolsExecution?: IPreToolsExecutionProcessor[];
        postToolExecution?: IPostToolExecutionProcessor[];
        onStreamingChunk?: IOnStreamingChunkProcessor[];
    };
    /**
     * Signal to abort the completion request.
     */
    abortSignal?: AbortSignal;
    /**
     * An optional identifier for the completion with tools call. This can be used for logging
     * and tracing purposes.
     */
    callId?: string;
    /**
     * The reasoning effort level for the model to use, if supported by the client.
     */
    reasoningEffort?: ReasoningEffort;
};

/** Gets the list of available custom agents. */
export declare function getCustomAgents(authInfo: AuthInfo, workingDir: string, integrationId?: string, logger?: RunnerLogger, settings?: RuntimeSettings): Promise<SweCustomAgent[]>;

/** Represents the GitHub CLI authentication information. */
declare type GhCliAuthInfo = {
    readonly type: "gh-cli";
    readonly host: string;
    readonly login: string;
    readonly token: string;
};

declare interface HandoffProgress {
    step: HandoffStep;
    status: "in-progress" | "complete";
    message?: string;
}

declare type HandoffStep = "load-session" | "validate-repo" | "check-changes" | "checkout-branch" | "create-session" | "save-session";

/** Represents the HMAC-based authentication information. */
declare type HMACAuthInfo = {
    readonly type: "hmac";
    readonly host: "https://github.com";
    readonly hmac: string;
};

export declare type HookEndEvent = z.infer<typeof HookEndEventSchema>;

/**
 * Hook invocation completes
 */
declare const HookEndEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"hook.end">;
    data: z.ZodObject<{
        hookInvocationId: z.ZodString;
        hookType: z.ZodString;
        output: z.ZodUnknown;
        success: z.ZodBoolean;
        error: z.ZodOptional<z.ZodObject<{
            message: z.ZodString;
            stack: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            message: string;
            stack?: string | undefined;
        }, {
            message: string;
            stack?: string | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    }, {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    };
    id: string;
    type: "hook.end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    };
    id: string;
    type: "hook.end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type HookStartEvent = z.infer<typeof HookStartEventSchema>;

/**
 * Hook invocation begins
 */
declare const HookStartEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"hook.start">;
    data: z.ZodObject<{
        hookInvocationId: z.ZodString;
        hookType: z.ZodString;
        input: z.ZodUnknown;
    }, "strip", z.ZodTypeAny, {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    }, {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    };
    id: string;
    type: "hook.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    };
    id: string;
    type: "hook.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

declare interface HTTPTransportConfig {
    type: "http";
    url: string;
    headers?: Record<string, string>;
    authProvider?: OAuthClientProvider;
}

/**
 * This event is temporary until we extract vision support from being internal to getCompletionWithTools.
 */
declare type ImageProcessingEvent = {
    kind: "image_processing";
    turn: number;
    imageProcessingMetrics: ImageProcessingMetrics;
};

declare type ImageProcessingMetrics = ({
    imagesExtractedCount: number;
    base64ImagesCount: number;
    imagesRemovedDueToSize: number;
    imagesRemovedDueToDimensions: number;
    imagesResized: number;
    imagesResolvedFromGitHubMCPCount: number;
    allImagesSendToLlm?: number;
} & Record<string, number>) | Record<string, never>;

/**
 * This event is temporary until we extract vision support from being internal to getCompletionWithTools.
 */
declare type ImageRemovalEvent = {
    kind: "images_removed";
    turn: number;
    largeImagesRemoved?: number;
    imagesRemoved: number;
};

/**
 * Configuration for infinite sessions with automatic context compaction and workspace persistence.
 * When enabled, sessions automatically manage context window limits through background compaction
 * and persist state to a workspace directory.
 */
declare interface InfiniteSessionConfig {
    /**
     * Whether infinite sessions are enabled.
     * @default true
     */
    enabled?: boolean;
    /**
     * Context utilization threshold (0.0-1.0) at which background compaction starts.
     * Compaction runs asynchronously, allowing the session to continue processing.
     * @default 0.80
     */
    backgroundCompactionThreshold?: number;
    /**
     * Context utilization threshold (0.0-1.0) at which the session blocks until compaction completes.
     * This prevents context overflow when compaction hasn't finished in time.
     * @default 0.95
     */
    bufferExhaustionThreshold?: number;
}

/**
 * A message to be injected into the conversation history after tool execution.
 * Used by tools like skills that need to add content to the conversation.
 */
declare type InjectedUserMessage = {
    /**
     * The content to inject as a user message.
     */
    content: string;
    /**
     * A source identifier for tracking where this message came from.
     * Used for filtering in the timeline display.
     * Example: "skill-pdf", "skill-code-reviewer"
     */
    source: string;
};

/**
 * In-memory transport that directly calls server methods without serialization.
 * This transport is used for servers that run in the same process as the client.
 */
declare class InMemoryClientTransport implements Transport {
    private server;
    onmessage?: (message: JSONRPCMessage) => void;
    onerror?: (error: Error) => void;
    onclose?: () => void;
    private serverTransport?;
    private closed;
    constructor(server: McpServer);
    /**
     * Set up bidirectional connection between client and server transports
     */
    start(): Promise<void>;
    send(message: JSONRPCMessage): Promise<void>;
    receive(message: JSONRPCMessage): void;
    close(): Promise<void>;
}

declare interface InMemoryTransportConfig {
    type: "memory";
    server: McpServer;
}

declare class InProcMCPTransport extends MCPTransport<ClientInfo> {
    private toolIdToClientInfo;
    constructor(settings: RuntimeSettings, logger: RunnerLogger, cacheProviderTools?: boolean);
    /**
     * Hook called when a provider (MCP client) is being refreshed.
     * Clears the tool ID mappings for this client.
     */
    protected onProviderRefresh(clientInfo: ClientInfo): Promise<void>;
    protected doInvokeTool(toolId: string, toolParams: any, toolCallId?: string, retryCount?: number): Promise<InvokeToolResponseData>;
    protected getProviderCacheKey(provider: ClientInfo): string;
    protected loadToolsFromProvider(clientInfo: ClientInfo): Promise<Record<string, BasicToolConfig>>;
    private static getToolIdFromClientAndToolName;
    private static getToolNameFromIdAndClientName;
}

/**
 * An installed plugin entry in user/project config.
 */
declare interface InstalledPlugin {
    /** Plugin name */
    name: string;
    /** Marketplace the plugin came from */
    marketplace: string;
    /** Version installed (if available) */
    version?: string;
    /** Installation timestamp */
    installed_at: string;
    /** Whether the plugin is currently enabled */
    enabled: boolean;
    /** Path where the plugin is cached locally */
    cache_path?: string;
}

export declare const INTEGRATION_ID: string;

export declare namespace internal {
    export {
        scoreSessionRelevance,
        HandoffStep,
        HandoffProgress,
        RelevanceScore,
        ScoredSessionMetadata,
        RELEVANCE_LABELS,
        LocalSessionManager
    }
}

declare type InvokeToolResponseData = {
    isToolError: boolean;
    content: any[];
};

declare interface IOnRequestErrorProcessor extends IToJson {
    /**
     * Called before an error is rethrown by the client. The processor may modify
     * the error in place.
     */
    preErrorThrow(error: unknown): Promise<void>;
    /**
     * Called when a request to the model fails. The processor should not modify
     * the error.
     */
    onRequestError(context: OnRequestErrorContext): Promise<OnRequestErrorResult | void>;
}

declare interface IOnStreamingChunkProcessor extends IToJson {
    /**
     * Called when a streaming chunk is received.
     */
    onStreamingChunk(context: StreamingChunkContext): void;
}

declare interface IPostToolExecutionProcessor extends IToJson {
    /**
     * Called after a tool has been executed. The processor should not
     * modify the tool result.
     */
    postToolExecution(context: PostToolExecutionContext): Promise<void>;
}

declare interface IPreRequestProcessor extends IToJson {
    /**
     * Called before a request (including retries of requests) is made to the model.
     *
     * - Any {@link Event}s emitted by this method will be re-emitted by the completion with tools call.
     */
    preRequest(context: PreRequestContext): AsyncGenerator<Event_2>;
}

declare interface IPreToolsExecutionProcessor extends IToJson {
    /**
     * Called before any tool calls are executed.
     */
    preToolsExecution(context: PreToolsExecutionContext): Promise<PreToolsExecutionResult>;
}

/**
 * The complete settings type that represents all possible configuration options.
 */
declare interface IRuntimeSettings {
    version: string;
    clientName: string;
    github: {
        serverUrl: string;
        uploadsUrl: string;
        downloadsUrl: string;
        secretScanningUrl: string;
        host: string;
        hostProtocol: string;
        token: string;
        user: {
            name: string;
            email: string;
            actorId?: number;
            actorLogin?: string;
        };
        owner: {
            id: number;
            name: string;
        };
        repo: {
            id: number;
            name: string;
            branch: string;
            commit: string;
            readWrite: boolean;
        };
        pr: {
            commitCount?: number;
        };
    };
    problem: {
        statement: string;
        contentFilterMode?: ContentFilterMode;
        action: AgentAction;
        customAgentName?: string;
    };
    service: {
        instance: {
            id: string;
        };
        /**
         * - Options beyond `model` are currently only respected
         * when going through `RuntimeHarness` methods.
         * - The value of this {@link ClientOptions.model} is NOT
         * the name of a model, it is AgentName:ModelName.
         */
        agent: ClientOptions;
        /**
         * Settings for tools that are used by the agent. Refer to the
         * source/documentation for each tool for their specific settings.
         */
        tools: {
            [toolName: string]: {
                [key: string]: unknown;
            };
        };
        callback: {
            url: string;
        };
    };
    api: {
        aipSweAgent: {
            token: string;
        };
        anthropic: {
            key: string;
            baseUrl: string;
        };
        openai: {
            baseUrl: string;
            apiKey: string;
            azureKeyVaultUri: string;
            azureSecretName: string;
            azure: {
                url: string;
                apiVersion: string;
                /** Bearer token for Azure AD authentication. When set, used instead of API key or managed identity. */
                bearerToken?: string;
            };
        };
        copilot: {
            url: string;
            integrationId: string;
            hmacKey: string;
            azureKeyVaultUri: string;
            token: string;
            useSessions: boolean;
            useAsyncSessions: boolean;
            sessionId: string;
            previousSessionIds: string[];
            /**
             * The W3C Trace Context traceparent header value for distributed tracing.
             * Format: {version}-{trace-id}-{parent-id}-{trace-flags}
             * Example: 00-0af7651916cd43dd8448eb211c80319c-b7ad6b7169203331-01
             */
            traceParent: string;
        };
        github: {
            /**
             * The GITHUB_PERSONAL_ACCESS_TOKEN that is passed to `github-mcp-server` when it is
             * started. (Currently only supported in the `cpd` entry point.)
             */
            mcpServerToken: string;
        };
    };
    blackbird: {
        mode: "initial-search" | "tool";
        backfillScoreThreshold?: number;
        repoNwo?: string;
        /**
         * The auth object contains the credentials for Blackbird's Metis API.
         * - modelBasedRetrievalToken: Token for model-based retrieval.
         * - metisApiKey: API key for Metis.
         */
        auth: {
            modelBasedRetrievalToken: string;
            metisApiKey: string;
        };
    };
    swebench_base_commit?: string;
    trajectory: {
        outputFile: string;
    };
    logs: {
        eventsLogDir: string;
    };
    job: {
        nonce?: string;
        eventType?: string;
    };
    onlineEvaluation: {
        disableOnlineEvaluation?: boolean;
        enableOnlineEvaluationOutputFile?: boolean;
    };
    tools: {
        bash: {
            /**
             * The default timeout for bash commands in seconds. If undefined, a default of 120 seconds (2 minutes) is used.
             */
            defaultTimeout?: number;
        };
        /**
         * Settings shared by all validation tools.
         */
        validation?: {
            /**
             * The shared timeout budget for all the validation tools in seconds. If undefined, a default of 180 seconds is used.
             * This timeout is shared across all validation tools, and once the total time spent exceeds this budget, no further validation tools will be run.
             * A validation tool will be cancelled if it is in progress when the budget is exceeded.
             */
            timeout?: number;
        };
        /**
         * Settings for handling large tool outputs.
         */
        largeOutput?: {
            /**
             * Maximum size in bytes before output is written to a temp file. Default is 50KB.
             */
            maxSizeBytes?: number;
            /**
             * Directory to write temp files to. Default is os.tmpdir().
             */
            outputDir?: string;
        };
    };
    /**
     * The set of feature flags passed to the agent runtime process by sweagentd.
     *
     * Only flags listed in internal/launcher/runtime_feature_flags.go are passed
     * to the runtime.
     *
     * To add a new flag:
     * - Define it in accordance with the feature flag docs: https://thehub.github.com/epd/engineering/products-and-services/dotcom/features/feature-flags/
     * - Add it to runtime_feature_flags.go
     * - Check whether it exists in the following object.
     *
     * Read a feature flag value with: @see isFeatureFlagEnabled
     *
     *    if (isFeatureFlagEnabled(settings, 'copilot_swe_agent_flag_name')) {
     *    }
     *
     * Report feature flag values in telemetry with: @see featureFlagsAsString in
     * a property named @see FEATURE_FLAGS_TELEMETRY_PROPERTY_NAME.
     *
     * NOTE: feature flag names may be visible to the user in logs or other output.
     */
    featureFlags: {
        [key: string]: boolean;
    };
    /**
     * EXP experiment configuration
     */
    experiments: {
        [key: string]: string;
    };
    /**
     * How many ms the runtime/the thing hosting the runtime has available to run
     * before it is considered to have timed out.
     */
    timeoutMs: number;
    /**
     * The time when the runtime/the thing hosting the runtime started, in ms since epoch.
     * May not be 100% accurate. Not typically set by hand.
     */
    startTimeMs: number;
    /**
     * Custom configuration directory for the session.
     * When set, overrides the default Copilot config directory (~/.copilot or $XDG_CONFIG_HOME/.copilot).
     */
    configDir: string;
}

/**
 * Returns true if the DEBUG or COPILOT_AGENT_DEBUG environment variable is set to 1 or true (case-insensitive).
 * If additionalVariables are provided, they are also checked.
 * @param additionalVariables Additional environment variables to check for debug logging.
 */
export declare function isDebugEnvironment(...additionalVariables: string[]): boolean;

/**
 * Something which must have an implementation of `toJSON()`. This can be used
 * for classes whose instances will likely be used with `JSON.stringify()` to avoid
 * any issues with stringification such as circular references or non-enumerable properties.
 *
 * More information on `toJSON()`: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/JSON/stringify#:~:text=If%20the%20value%20has%20a%20toJSON()%20method%2C%20it%27s%20responsible%20to%20define%20what%20data%20will%20be%20serialized.
 */
declare interface IToJson {
    toJSON(): string;
}

declare type LargeOutputOptions = {
    maxOutputSizeBytes: number;
    outputDir?: string;
};

/**
 * Configuration for handling large tool outputs.
 */
declare interface LargeToolOutputConfig {
    /**
     * Whether large output handling is enabled. Default is true.
     */
    enabled?: boolean;
    /**
     * Maximum size in bytes before output is written to a temp file. Default is 50KB.
     */
    maxSizeBytes?: number;
    /**
     * Directory to write temp files to. Default is os.tmpdir().
     */
    outputDir?: string;
}

/**
 * Session class with event sourcing
 */
export declare class LocalSession extends Session<LocalSessionMetadata> {
    private callback;
    private isProcessing;
    private itemQueue;
    private premiumRequestProcessor;
    private immediatePromptProcessor;
    private compactionProcessor;
    private abortController?;
    private queuedCommandHandler;
    private activeCustomAgents;
    mcpHostCache: McpHostCache;
    private modelListCache;
    private warnedUnknownTools;
    private sessionWorkspace;
    private workspaceEnabled;
    private lastTodoContent;
    private lastPlanUpdateTurn;
    private currentTurn;
    private static readonly PLAN_REMINDER_TURN_THRESHOLD;
    private cachedRepoMemories;
    private compactionCancelled;
    private manualCompactionAbortController;
    /**
     * Creates a new Session instance.
     *
     * In practice, use SessionManager.createSession() to create sessions and SessionManager.getSession() / SessionManager.getLastSession() to retrieve existing sessions.
     *
     * @param options - Configuration options for the session including model provider, tools, hooks, environment settings, and metadata (sessionId, startTime, modifiedTime). If metadata is not provided, new values are generated.
     */
    constructor(options?: SessionOptions);
    /**
     * Normalize infiniteSessions config from boolean or object to a full config object.
     */
    private normalizeInfiniteSessionsConfig;
    /**
     * Update workspace summary from user message content.
     */
    private updateWorkspaceSummary;
    /**
     * Initialize workspace - load existing or create new.
     * Workspaces are always created when infinite sessions are enabled.
     */
    private initializeWorkspace;
    /**
     * Update the workspace context based on current workspace state.
     * This context is used in system prompts to inform the agent about workspace files.
     */
    private updateWorkspaceContext;
    /**
     * Updates session options after creation.
     * This method allows selectively updating configuration options without recreating the session.
     * Only the provided options will be updated; omitted options remain unchanged.
     *
     * @param options - Partial session options to update
     *
     * @example
     * ```typescript
     * // Update multiple options at once
     * session.updateOptions({
     *   logger: fileLogger,
     *   mcpServers: mcpConfig,
     *   customAgents: loadedAgents
     * });
     *
     * // Or use convenience methods for single updates
     * session.setAuthInfo(newAuthInfo); // Preferred for auth
     * session.setSelectedModel(newModel); // Preferred for model
     * ```
     */
    updateOptions(options: Partial<UpdatableSessionOptions>): void;
    getMetadata(): LocalSessionMetadata;
    /**
     * Get the current workspace, if any.
     */
    getWorkspace(): Workspace | null;
    /**
     * Check if workspace features are enabled.
     */
    isWorkspaceEnabled(): boolean;
    /**
     * Get the workspace path for this session.
     * Returns null if workspace features are not enabled.
     * Returns the path even if workspace.yaml doesn't exist yet (for prompt context).
     */
    getWorkspacePath(): string | null;
    /**
     * Get the number of checkpoints (compaction summaries) in the workspace.
     */
    getCheckpointCount(): number;
    /**
     * Check if the session already has a title (user-provided name or AI-generated summary).
     */
    private hasSessionTitle;
    /**
     * List checkpoints with their titles for context injection.
     */
    listCheckpointTitles(): Promise<{
        number: number;
        title: string;
        filename: string;
    }[]>;
    /**
     * Check if a plan.md file exists in the workspace.
     */
    hasPlan(): boolean;
    /**
     * Get plan.md content for post-compaction message.
     * Returns null if workspace is not enabled or plan doesn't exist.
     */
    private getPlanContentForCompaction;
    /**
     * Update the last todo content (called when update_todo tool is used).
     * This content will be included in post-compaction messages.
     */
    setLastTodoContent(content: string | null): void;
    /**
     * Get the last todo content.
     */
    getLastTodoContent(): string | null;
    /**
     * Check if a plan update reminder should be shown.
     * Returns true if:
     * - Workspace is enabled
     * - Plan.md hasn't been updated in the last N turns
     * - A plan.md file exists (we only remind to update, not create)
     */
    shouldShowPlanReminder(): boolean;
    /**
     * Get the plan reminder message to inject into user prompts.
     * Returns null if no reminder should be shown.
     */
    getPlanReminderMessage(): string | null;
    /**
     * Mark plan as recently updated (resets the reminder timer).
     * Called when plan.md is detected to be written.
     */
    markPlanUpdated(): void;
    /**
     * Emit telemetry when agent writes to a workspace file (plan.md or files/).
     */
    private emitWorkspaceFileTelemetry;
    /**
     * Emit telemetry when agent reads a workspace file (plan.md, checkpoints, or files/).
     */
    private emitWorkspaceFileReadTelemetry;
    /**
     * Ensure workspace exists for this session.
     * Creates workspace.yaml and directory structure if needed.
     */
    ensureWorkspace(context?: WorkspaceContext): Promise<Workspace>;
    /**
     * Persist a compaction summary as a checkpoint.
     * Called automatically when compaction completes.
     */
    private persistCompactionCheckpoint;
    /**
     * Truncate workspace checkpoints to align with the current session history.
     * Used after rollback to remove compaction checkpoints created after the rollback point.
     */
    truncateWorkspaceCheckpoints(keepCount: number): Promise<void>;
    /**
     * Sends a message to the session and executes the agentic loop.
     * Messages can be queued or sent immediately during an ongoing turn.
     *
     * @param options - Send options including prompt, attachments, and mode
     * @param options.prompt - The prompt text to send
     * @param options.attachments - Optional file/directory attachments
     * @param options.mode - "enqueue" (default) adds to queue and processes when ready, "immediate" injects during current turn
     * @param options.abortController - Optional AbortController to abort the send operation
     * @returns A Promise that resolves when the message has been queued or processed
     *
     * @example
     * ```typescript
     * // Send a message (default enqueue mode)
     * session.send({
     *   prompt: "What files are in this directory?",
     *   attachments: [{ type: "directory", path: "/path/to/dir" }]
     * });
     *
     * // Send immediate message during processing
     * session.send({
     *   prompt: "Continue with that approach",
     *   mode: "immediate"
     * });
     * ```
     */
    send(options: SendOptions): Promise<void>;
    /**
     * Core logic for adding an item to the queue.
     *
     * @param item - The queued item (either a command or message)
     * @param prepend - If true, adds to the front of the queue (for priority messages)
     */
    private addItemToQueue;
    /**
     * Enqueue any item (command or message) to be processed after the current agentic work completes.
     * Items are processed in FIFO order.
     *
     * If the session is not currently processing, this will also kick off queue processing.
     * This ensures that items transferred from another session (e.g., after /clear)
     * actually get executed even if there are no messages to trigger processing.
     *
     * @param item - The queued item (either a command or message)
     */
    enqueueItem(item: QueuedItem): void;
    /**
     * Enqueue a slash command to be executed after the current agentic work completes.
     * Commands are processed in FIFO order alongside user messages.
     *
     * If the session is not currently processing, this will also kick off queue processing.
     * This ensures that commands transferred from another session (e.g., after /clear)
     * actually get executed even if there are no messages to trigger processing.
     *
     * @param command - The full command string including the slash, e.g., "/compact" or "/model gpt-4"
     */
    enqueueCommand(command: string): void;
    /**
     * Enqueue a user message item to be processed later.
     * This is a utility for creating properly typed message queue items.
     * Uses addItemToQueue internally for consistent mode normalization.
     *
     * @param options - The send options for the message
     * @param prepend - If true, adds to the front of the queue (for priority messages)
     */
    private enqueueUserMessage;
    /**
     * Set the handler for executing queued slash commands.
     * This should be called by the CLI to register its slash command execution logic.
     * Pass undefined to clear the handler (e.g., during cleanup).
     */
    setQueuedCommandHandler(handler: QueuedCommandHandler | undefined): void;
    /**
     * Process the item queue, handling both messages and commands in FIFO order.
     */
    private processQueue;
    abort(): Promise<void>;
    /**
     * Check if the session is currently in a state where it can be aborted.
     * Returns true if there's an active abort controller that hasn't been aborted yet.
     * This is important for queued operations where the CLI may not have direct access
     * to the abort controller (e.g., when messages are processed from the queue).
     */
    isAbortable(): boolean;
    /**
     * TODO(queueing-improvements)
     * If we support steering or queuing for remote sessions, these will need
     * to become async, but since we don't right now, it is MUCH easier for the UI
     * to have them be synchronous.
     */
    getPendingSteeringMessagesDisplayPrompt(): ReadonlyArray<string>;
    getPendingQueuedMessagesDisplayPrompt(): ReadonlyArray<string>;
    /**
     * Get pending queued items for UI display.
     * Returns both messages and commands with their display text.
     */
    getPendingQueuedItems(): ReadonlyArray<PendingQueuedItem>;
    /**
     * @deprecated Use getPendingQueuedItems() instead for proper display of mixed queue items.
     */
    getPendingQueuedMessages(): ReadonlyArray<string>;
    /**
     * Clear all pending steering and queued items (messages and commands).
     * Used internally when the agentic loop is aborted (e.g., user rejected a tool permission).
     */
    clearPendingItems(): void;
    /**
     * @deprecated Use clearPendingItems() instead.
     */
    clearPendingMessages(): void;
    /**
     * Compacts the conversation history into a single summary message.
     * This method is used by the /compact slash command for manual compaction.
     * Uses the same system message and tools as the core agent loop for consistency.
     *
     * @returns Promise that resolves with compaction results
     * @throws Error if compaction fails or prerequisites aren't met
     */
    compactHistory(): Promise<CompactionResult>;
    /**
     * Process queued items (messages and commands) through the agentic loop.
     * Used by send() and after manual compaction completes.
     * No-op if already processing or queue is empty.
     */
    private processQueuedItems;
    /**
     * Handle background compaction completion callback.
     * This is called immediately when background compaction finishes, allowing the session
     * to emit events and update state without waiting for the next preRequest call.
     */
    private handleCompactionComplete;
    /**
     * Cancels any in-progress background compaction.
     *
     * This should be called when the session state is being rolled back (e.g., Esc Esc),
     * to prevent the compaction result from being applied after the rollback.
     */
    cancelBackgroundCompaction(): void;
    abortManualCompaction(): boolean;
    /**
     * Initialize MCP host if configured
     */
    private initializeMcpHost;
    /**
     * Get connected IDE info if available.
     * Returns undefined if no IDE is connected or if the MCP host is not a CliMcpHost.
     */
    private getConnectedIdeInfo;
    /**
     * Initializes the session and validates tool filter configuration.
     * This method should be called after the session is fully configured (auth, model, MCP servers)
     * but before the first message is sent. It eagerly builds and caches the tool definitions and system message
     * so they are available for features like /context that need them before the first message.
     * It will also emit warnings for any unknown tool names specified in availableTools or excludedTools.
     *
     * @returns Promise that resolves when initialization and validation is complete
     */
    initializeAndValidateTools(): Promise<void>;
    /**
     * Shared method to build settings and initialize tools.
     * Used by both initializeAndValidateTools() and runAgenticLoop().
     *
     * @param problemStatement - Optional problem statement for settings (used by runAgenticLoop)
     * @returns Settings and tools, or undefined if initialization failed
     */
    private buildSettingsAndTools;
    private isToolEnabled;
    /**
     * Validates tool filter configuration and emits info about disabled tools and warnings for unknown tool names.
     */
    private validateToolFilters;
    /**
     * Builds custom agent tools from the available custom agents.
     *
     * @param customAgents - Array of custom agents to convert to tools
     * @param parentTools - Tools available to the custom agents
     * @param runtimeContext - Runtime context for custom agent execution
     * @param mcpServerProvider - Optional MCP server provider for custom agent MCP servers
     * @returns Promise that resolves to an array of custom agent tools
     */
    private buildCustomAgentTools;
    /**
     * Filters tools based on the selected custom agent, if any.
     *
     * @param allTools - All available tools
     * @returns Filtered tools based on selected custom agent restrictions
     */
    private filterToolsForSelectedAgent;
    private invokeCallbacks;
    private getModelList;
    /**
     * Creates a client instance with current session configuration.
     * Extracted from runAgenticLoop to allow reuse for standalone LLM calls.
     *
     * @returns Promise that resolves to a configured client and settings
     * @throws Error if session was not created with authentication info or custom provider
     */
    private getClient;
    /**
     * Generates a summarized version of the conversation context suitable for delegation.
     * Uses an LLM to create a concise summary of the existing conversation that fits
     * within size constraints (20k characters).
     *
     * @returns Promise that resolves to a markdown summary of the session context
     * @throws Error if session was not created with authentication info or if summarization fails
     */
    getContextSummary(): Promise<string>;
    /**
     * Executes the full agentic loop for a given prompt.
     * This method orchestrates the complete AI agent workflow including:
     * - Running hooks (userPromptSubmitted, sessionStart, preToolUse, postToolUse, sessionEnd)
     * - Building and sending the prompt to the language model
     * - Processing model responses and tool calls
     * - Executing tools and feeding results back to the model
     * - Emitting events throughout the process
     *
     * This is the core method that powers the `send()` functionality.
     * Most users should call `send()` instead, which handles queuing and mode selection.
     *
     * @param prompt - The user's prompt/instruction text
     * @param attachments - Optional array of file or directory attachments to include with the prompt
     * @returns A Promise that resolves when the agent loop completes
     * @throws Error if the session was not created with authentication info/custom provider or model
     */
    private runAgenticLoop;
}

/**
 * SessionManager subclass that persists sessions to JSONL files.
 * Uses SessionEventState for all file system operations.
 */
declare class LocalSessionManager implements SessionManager<LocalSessionMetadata, LocalSession> {
    private sessionWriters;
    private copilotVersion;
    private flushDebounceMs;
    private settings?;
    constructor({ version, flushDebounceMs, settings }?: LocalSessionManagerOptions);
    /**
     * Get effective settings, preferring options.configDir over this.settings
     */
    private getEffectiveSettings;
    /**
     * Create a new file-backed session
     */
    createSession(options?: SessionOptions, emitStart?: boolean): Promise<LocalSession>;
    private loadHooks;
    /**
     * Get existing session by ID
     * ALWAYS loads from disk to ensure freshness
     * Supports both JSONL and legacy JSON formats
     *
     * @throws Error if the session exists but cannot be parsed (e.g., unknown event types, corrupt data)
     * @returns The session, or undefined if the session does not exist
     */
    getSession(options: SessionOptions & {
        sessionId: string;
    }, resume?: boolean): Promise<LocalSession | undefined>;
    /**
     * Get the most recently updated session
     */
    getLastSession(options?: Omit<SessionOptions, "sessionId">): Promise<LocalSession | undefined>;
    /**
     * Get the ID of the most recently updated session
     */
    getLastSessionId(): Promise<string | undefined>;
    saveSession(session: Session): Promise<void>;
    private loadSession;
    /**
     * Load a legacy JSON session and convert it to LocalSession
     * This creates a session.import_legacy event to preserve the legacy data
     */
    private loadLegacySession;
    /**
     * List all sessions by reading from disk
     */
    listSessions(): Promise<LocalSessionMetadata[]>;
    /**
     * List sessions sorted by relevance to the current working directory context.
     * Sessions matching the current repo+branch appear first, followed by same repo,
     * then same gitRoot or cwd, then all others sorted by time.
     *
     * @param currentContext The current working directory context to compare against
     */
    listSessionsSortedByRelevance(currentContext?: WorkingDirectoryContext): Promise<LocalSessionMetadata[]>;
    /**
     * List sessions with their relevance scores for grouping in the UI.
     * Sessions are sorted by score (descending), then by modifiedTime (descending).
     *
     * @param currentContext The current working directory context to compare against
     */
    listSessionsWithScores(currentContext?: WorkingDirectoryContext): Promise<ScoredSessionMetadata<LocalSessionMetadata>[]>;
    /**
     * Delete a session from disk
     */
    deleteSession(sessionId: string): Promise<void>;
    /**
     * Close session - flushes pending data and cleans up resources but keeps session in memory
     */
    closeSession(sessionId: string): Promise<void>;
    /**
     * Get the sessions directory path (for debugging/logging)
     */
    getSessionsDirectory(): string;
    /**
     * Handoff a remote session to local by validating repository, checking git state,
     * and creating a local session with the remote session's events.
     * This is an async generator that yields progress updates.
     *
     * @param remoteSession - The remote session to handoff
     * @param sessionOptions - Optional session options to pass to createSession
     * @yields HandoffProgress updates for each step
     * @returns A new local session with the same events
     * @throws Error if validation fails or git operations fail
     */
    handoffSession(remoteSession: Session, sessionOptions?: SessionOptions): AsyncGenerator<HandoffProgress, LocalSession, undefined>;
    /**
     * Helper method to fetch from git remote
     */
    private gitFetch;
}

declare type LocalSessionManagerOptions = {
    version?: string;
    flushDebounceMs?: number;
    settings?: RuntimeSettings;
};

export declare interface LocalSessionMetadata extends SessionMetadata {
    readonly isRemote: false;
}

/** Basic logging interface */
declare interface Logger {
    info(message: string): void;
    debug(message: string): void;
    warning(message: string): void;
    error(message: string): void;
}

export declare enum LogLevel {
    None = 0,
    Error = 1,
    Warning = 2,
    Info = 3,
    Debug = 4,
    All = 4,// Logs everything
    Default = 3
}

declare type LogLevel_2 = keyof Logger;

/** Interface for an implementer of writing log messages asynchronously. */
declare interface LogWriter {
    /** Write a log message at the given level.
     * @returns A promise that resolves when all pending writes are complete
     */
    writeLog(level: LogLevel_2, message: string): Promise<void>;
    /** @returns the current file path or other identifier if not writing to a file. */
    outputPath(): string;
}

/**
 * Manages the lifecycle of MCP (Model Context Protocol) servers and provides access to their tools.
 */
declare class McpHost {
    protected logger: RunnerLogger;
    protected onOAuthRequired?: ((serverName: string, serverUrl: string, staticClientConfig?: {
        clientId: string;
        publicClient?: boolean;
    }) => Promise<OAuthClientProvider | undefined>) | undefined;
    protected registry: MCPRegistry;
    private processor;
    protected config: MCPServersConfig;
    private startServersPromise?;
    protected transport: InProcMCPTransport | null;
    private disabledServers;
    private progressCallback?;
    constructor(logger: RunnerLogger, mcpConfig: string | MCPServersConfig, disabledServers?: string[], envValueMode?: EnvValueMode, onOAuthRequired?: ((serverName: string, serverUrl: string, staticClientConfig?: {
        clientId: string;
        publicClient?: boolean;
    }) => Promise<OAuthClientProvider | undefined>) | undefined);
    /**
     * Called when a server sends a tools/list_changed notification
     */
    private handleToolsChanged;
    startServers(): Promise<void>;
    /**
     * Extension point for subclasses to inject default servers into the config.
     * This method is called during startServers() before processing the servers.
     * Subclasses should override this method and mutate the config parameter in place
     * to add custom server configurations.
     */
    protected injectDefaultServers(_config: MCPServersConfig): Promise<void>;
    private processServersWithExtensions;
    stopServers(): Promise<void>;
    /**
     * Gets all available tools from the MCP servers. Starts the servers with @see startServers if they have not already been started.
     * the tools returned should not be used after @see stopServers has been called.
     *
     * @param settings - The runtime settings.
     * @param logger - The logger instance.
     * @param permissions - Permissions configuration for tool access.
     * @returns A promise that resolves to an array of tools.
     */
    getTools(settings: RuntimeSettings, logger: RunnerLogger, permissions: PermissionsConfig): Promise<Tool_2[]>;
    /**
     * Gets the current MCP configuration.
     */
    getConfig(): MCPServersConfig;
    /**
     * Gets all connected MCP clients.
     * @returns A record of server names to their client instances
     */
    getClients(): Record<string, Client>;
    /**
     * Start a single MCP server with the given configuration
     * @param serverName - Unique name for the server
     * @param config - Server configuration
     * @returns Promise that resolves when server is started
     */
    startServer(serverName: string, config: MCPServerConfig): Promise<void>;
    /**
     * Stop a single MCP server
     * @param serverName - Name of the server to stop
     * @returns Promise that resolves when server is stopped
     */
    stopServer(serverName: string): Promise<void>;
    /**
     * Restart a server with new configuration
     * @param serverName - Name of the server to restart
     * @param config - New server configuration
     * @returns Promise that resolves when server is restarted
     */
    restartServer(serverName: string, config: MCPServerConfig): Promise<void>;
    /**
     * Check if a server is currently running
     * @param serverName - Name of the server to check
     * @returns True if server is running
     */
    isServerRunning(serverName: string): boolean;
    /**
     * Check if a server is disabled
     * @param serverName - Name of the server to check
     * @returns True if server is disabled
     */
    isServerDisabled(serverName: string): boolean;
    /**
     * Disable a server at runtime (does not persist across sessions)
     * @param serverName - Name of the server to disable
     * @returns Promise that resolves when server is disabled
     */
    disableServer(serverName: string): Promise<void>;
    /**
     * Enable a previously disabled server at runtime
     * @param serverName - Name of the server to enable
     * @returns Promise that resolves when server is enabled
     */
    enableServer(serverName: string): Promise<void>;
    /**
     * Extension point for subclasses to start a built in server that is not listed in
     * the MCP Config.
     * @param serverName - Name of the server to enable
     * @returns Promise that resolves when server handling is complete
     */
    protected startBuiltInServer(_serverName: string): Promise<void>;
    /**
     * Get the configuration for a running server
     * @param serverName - Name of the server
     * @returns Server configuration or undefined if not found
     */
    getServerConfig(serverName: string): MCPServerConfig | undefined;
    /**
     * Set the callback to be invoked when an MCP tool reports progress.
     * @param callback - The callback to invoke with (toolCallId, progressMessage)
     */
    setProgressCallback(callback: ToolProgressCallback | undefined): void;
}

/**
 * Cache that maintains a mapping of agent ID to MCP host.
 * The root agent uses an empty string as its ID.
 */
declare class McpHostCache {
    private hosts;
    private logger;
    constructor(logger: RunnerLogger);
    /**
     * Get or create an MCP host for the given agent ID.
     * Returns undefined if mcpServers is empty or undefined.
     */
    getOrCreateHost(agentId: string, mcpServers: Record<string, MCPServerConfig> | undefined): Promise<McpHost | undefined>;
    /**
     * Get an existing host for the given agent ID.
     */
    getHost(agentId: string): McpHost | undefined;
    /**
     * Stop and remove all MCP hosts.
     */
    cleanup(): Promise<void>;
    /**
     * Get the number of hosts in the cache.
     */
    size(): number;
}

declare interface MCPInMemoryServerConfig extends MCPServerConfigBase {
    type: "memory";
    serverInstance: McpServer;
}

declare interface MCPLocalServerConfig extends MCPServerConfigBase {
    type?: "local" | "stdio";
    command: string;
    args: string[];
    /**
     * An object of the environment variables to pass to the server.
     *
     * The interpretation of this object depends on the environment variable mode:
     * - In 'indirect' mode (default): Hybrid approach for backward compatibility
     *   - If value contains $ or ${...}, tries variable substitution first
     *   - If substitution succeeds (changes the value), uses the resolved value
     *   - Otherwise, treats value as the name of an env var to read from current process
     *   Example: { "FOO": "BAR" } sets FOO=process.env.BAR (legacy)
     *   Example: { "FOO": "$BAR" } sets FOO=process.env.BAR (variable expansion)
     *   Example: { "FOO": "${BAR:-default}" } sets FOO=process.env.BAR or "default"
     * - In 'direct' mode: Key is the env var name to set in the MCP server,
     *   Value is the literal value to set. Supports variable expansion:
     *   - $VAR or ${VAR}: expands to process.env.VAR
     *   - ${VAR:-default}: expands to process.env.VAR, or "default" if VAR is undefined
     *   Example: { "FOO": "bar" } sets FOO=bar
     *   Example: { "FOO": "${BAR}" } or { "FOO": "$BAR" } sets FOO=process.env.BAR
     *   Example: { "FOO": "${BAR:-fallback}" } sets FOO=process.env.BAR or "fallback"
     *   Example: { "URL": "https://${HOST}:${PORT}" } expands both variables
     *
     * Empty means no env vars passed.
     */
    env?: Record<string, string>;
    cwd?: string;
}

/**
 * A permission request for invoking an MCP tool.
 */
declare type MCPPermissionRequest = {
    readonly kind: "mcp";
    /** The name of the MCP Server being targeted e.g. "github-mcp-server" */
    readonly serverName: string;
    /** The name of the tool being targeted e.g. "list_issues" */
    readonly toolName: string;
    /** The title of the tool being targeted e.g. "List Issues" */
    readonly toolTitle: string;
    /**
     * The _hopefully_ JSON arguments that will be passed to the MCP tool.
     *
     * This should be an object, but it's not parsed before this point so we can't guarantee that.
     * */
    readonly args: unknown;
    /**
     * Whether the tool is read-only (e.g. a `view` operation) or not (e.g. an `edit` operation).
     */
    readonly readOnly: boolean;
};

declare class MCPRegistry {
    private logger;
    private clientFactory;
    private transportFactory;
    envValueMode: EnvValueMode;
    clients: Record<string, Client>;
    transports: Record<string, Transport>;
    configs: Record<string, MCPServerConfig>;
    private toolsChangedCallback?;
    constructor(logger: RunnerLogger, clientFactory?: ClientFactory, transportFactory?: TransportFactory, envValueMode?: EnvValueMode);
    /**
     * Register a callback to be invoked when a server sends a tools/list_changed notification.
     * The callback can be sync or async.
     */
    setToolsChangedCallback(callback: (clientName: string) => void | Promise<void>): void;
    startLocalMcpClient(serverName: string, serverConfig: MCPLocalServerConfig): Promise<void>;
    startHttpMcpClient(serverName: string, serverConfig: MCPRemoteServerConfig, authProvider?: OAuthClientProvider): Promise<void>;
    startSseMcpClient(serverName: string, serverConfig: MCPRemoteServerConfig, authProvider?: OAuthClientProvider): Promise<void>;
    startInMemoryMcpClient(serverName: string, serverConfig: MCPServerConfig, serverInstance: McpServer): Promise<void>;
    /**
     * Test a connection to a remote MCP server without registering it.
     * This is useful for validating server configuration before saving.
     * Throws UnauthorizedError if the server requires authentication.
     * Throws other errors if the connection fails for other reasons.
     *
     * @param serverUrl - The URL of the remote server
     * @param serverType - The type of transport ("http" or "sse")
     * @param headers - Optional headers to include in requests
     * @param authProvider - Optional OAuth provider for authenticated connections
     */
    testRemoteConnection(serverUrl: string, serverType: "http" | "sse", headers?: Record<string, string>, authProvider?: OAuthClientProvider): Promise<void>;
    private setupAndConnectClient;
    getTools(mcpServersConfig: MCPServersConfig | undefined, sessionClient?: CopilotSessionsClient): Promise<Record<string, Tool>>;
    private getServerTools;
    private logServerSuccessWithTools;
}

declare interface MCPRemoteServerConfig extends MCPServerConfigBase {
    type: "http" | "sse";
    /**
     * URL of the remote server
     * NOTE: this has to be converted to a URL object before giving to transport.
     * TransportFactory will handle this conversion.
     */
    url: string;
    /**
     * Optional. HTTP headers to include in requests to the remote server.
     * This can be used for authentication or other purposes.
     * For example, you might include an Authorization header.
     */
    headers?: Record<string, string>;
    /**
     * Optional. OAuth client ID for pre-registered (static) OAuth clients.
     * When set, dynamic client registration is skipped and this client ID is used.
     * If not set, dynamic client registration (RFC 7591) is used when OAuth is detected.
     *
     * OAuth is automatically detected by probing the server at connection time
     * (via /.well-known/oauth-protected-resource or 401 Unauthorized responses).
     */
    oauthClientId?: string;
    /**
     * Optional. Indicates whether this is a public OAuth client (no secret).
     * Defaults to true (public client).
     * When false, the client secret is retrieved from the system keychain.
     */
    oauthPublicClient?: boolean;
}

declare type MCPServerConfig = MCPLocalServerConfig | MCPRemoteServerConfig | MCPInMemoryServerConfig;

declare interface MCPServerConfigBase {
    /**
     * Optional human-readable name for this server.
     *
     * This is intended to preserve original registry/config names that may contain "/".
     * The object key used to identify the server may be adapted (e.g. "/" -> "__") for
     * internal IDs, but this value should remain exactly as provided by the user/registry.
     */
    displayName?: string;
    /**
     * List of tools to include from this server. [] means none. "*" means all.
     */
    tools: string[];
    /**
     * Indicates "remote" or "local" server type.
     * If not specified, defaults to "local".
     */
    type?: string;
    /**
     * Optional. Denotes if this is a MCP server we have defined to be used when
     * the user has not provided their own MCP server config.
     *
     * Marked optional as configs coming from users will/should not have this set. Defaults to `false`.
     */
    isDefaultServer?: boolean;
    /**
     * Optional. Either a content filter mode for all tools from this server, or a map of tool name to content filter mode for the tool with that name.
     * If not specified, defaults to "hidden_characters"
     */
    filterMapping?: Record<string, ContentFilterMode> | ContentFilterMode;
    /**
     * Optional. Timeout in milliseconds for tool calls to this server.
     * If not specified, a default is used.
     */
    timeout?: number;
    /**
     * Optional. Name of the plugin that provided this MCP server.
     * Only set for servers loaded from installed plugins.
     */
    sourcePlugin?: string;
}

declare interface MCPServersConfig {
    mcpServers: Record<string, MCPServerConfig>;
}

declare abstract class MCPTransport<ToolsProviderT = unknown> {
    protected readonly settings: RuntimeSettings;
    protected readonly logger: RunnerLogger;
    protected readonly cacheProviderTools: boolean;
    private cachedTools;
    private _progressCallback?;
    constructor(settings: RuntimeSettings, logger: RunnerLogger, cacheProviderTools: boolean);
    /**
     * Set the callback to be invoked when an MCP tool reports progress.
     */
    setProgressCallback(callback: ToolProgressCallback | undefined): void;
    protected get progressCallback(): ToolProgressCallback | undefined;
    /**
     * Refresh tools for a specific provider.
     * This completely reloads the provider's tools, clearing all cached state.
     * Called when a tools/list_changed notification is received or when tools need to be refreshed.
     */
    refreshProvider(provider: ToolsProviderT): Promise<void>;
    /**
     * Hook for subclasses to perform additional cleanup when a provider is refreshed.
     * Default implementation does nothing.
     */
    protected onProviderRefresh(_provider: ToolsProviderT): Promise<void>;
    invokeTool(toolId: string, toolParams: any, filterMode?: ContentFilterMode, toolCallId?: string): Promise<ToolResultExpanded>;
    /**
     * Allows us to directly invoke a tool, which could be one that is suppressed from the tool list.
     * - Response can be either a text or a Base64 encoded image based on mcp-client/src/MCPServer.ts implementation of the invoke-tool endpoint.
     * - If `isToolError` is true, it means the tool call failed.
     */
    protected abstract doInvokeTool(toolId: string, toolParams: any, toolCallId?: string): Promise<InvokeToolResponseData>;
    protected invokeToolResponseToToolResult(responseData: InvokeToolResponseData, filterMode: ContentFilterMode): ToolResultExpanded;
    loadTools(provider: ToolsProviderT, permissions?: PermissionsConfig): Promise<Tool_2[]>;
    protected abstract getProviderCacheKey(provider: ToolsProviderT): string;
    protected abstract loadToolsFromProvider(provider: ToolsProviderT): Promise<Record<string, BasicToolConfig>>;
}

/**
 * A permission request for storing memory.
 */
declare type MemoryPermissionRequest = {
    readonly kind: "memory";
    /** The subject of the memory being stored */
    readonly subject: string;
    /** The fact being stored */
    readonly fact: string;
    /** The source citations for the fact */
    readonly citations: string;
};

/**
 * All types of message events that can be emitted by the `Client`.
 */
declare type MessageEvent_2 = {
    kind: "message";
    turn?: number;
    callId?: string;
    modelCall?: ModelCallParam;
    message: ChatCompletionMessageParam & ReasoningMessageParam;
} | AssistantMessageEvent_2 | UserMessageEvent_2 | ToolMessageEvent;

declare type Model = {
    id: string;
    name: string;
    capabilities: {
        supports: {
            vision: boolean;
        };
        limits: {
            max_prompt_tokens?: number;
            max_context_window_tokens: number;
            vision?: {
                supported_media_types: string[];
                max_prompt_images: number;
                max_prompt_image_size: number;
            };
        };
    };
    policy?: {
        state: "enabled" | "disabled" | "unconfigured";
        terms: string;
    };
    billing?: {
        multiplier: number;
    };
};

declare type ModelCallFailureEvent = {
    kind: "model_call_failure";
    turn: number;
    callId?: string;
    modelCallDurationMs: number;
    /**
     * The model call that failed, if available.
     */
    modelCall: ModelCallParam;
    /**
     * A string representation of the messages sent as input to the model call, if available.
     */
    requestMessages?: string;
};

/**
 * -----------------------------------------------------------------------
 * Events
 * -----------------------------------------------------------------------
 *
 * Event (Union Type)
 * ├── TurnStartedEvent
 * ├── TurnEndedEvent
 * ├── TurnFailedEvent
 * ├── TurnRetryEvent
 * ├── ModelCallSuccessEvent
 * ├── ModelCallFailureEvent
 * ├── ToolExecutionEvent
 * ├── ImageProcessingEvent
 * ├── ImageRemovalEvent
 * ├── TruncationEvent
 * ├── MessageEvent
 * │   ├── AssistantMessageEvent
 * │   ├── UserMessageEvent
 * │   └── ToolMessageEvent
 * ├── ResponseEvent
 * └── SessionLogEvent
 */
/**
 * All types of events that can be emitted by the `Client`.
 */
declare interface ModelCallParam {
    api_id?: string;
    model?: string;
    error?: string;
    status?: number;
    request_id?: string;
    initiator?: string;
}

declare type ModelCallSuccessEvent = {
    kind: "model_call_success";
    turn: number;
    callId?: string;
    modelCallDurationMs: number;
    /**
     * Time to first token in milliseconds. Only available for streaming requests.
     */
    ttftMs?: number;
    /**
     * Average inter-token latency in milliseconds. Only available for streaming requests.
     * Calculated as the average time between successive tokens.
     */
    interTokenLatencyMs?: number;
    modelCall: ModelCallParam;
    responseChunk: CopilotChatCompletionChunk;
    responseUsage: OpenAI.ChatCompletion["usage"];
    /**
     * A string representation of the messages sent as input to the model call, if available.
     */
    requestMessages?: string;
    quotaSnapshots?: Record<string, QuotaSnapshot>;
};

export declare class NoopLogger extends BaseLogger implements RunnerLogger {
    constructor();
    debug(_message: string): void;
    log(_message: string): void;
    info(_message: string): void;
    notice(_message: string | Error): void;
    warning(_message: string | Error): void;
    error(_message: string | Error): void;
    startGroup(_name: string, _level?: LogLevel): void;
    endGroup(_level?: LogLevel): void;
}

declare type OnRequestErrorContext = {
    /**
     * The current turn.
     */
    readonly turn: number;
    /**
     * The current retry attempt.
     */
    readonly retry: number;
    /**
     * The maximum number of retry attempts.
     */
    readonly maxRetries: number;
    /**
     * The error received in response to a request.
     */
    readonly error: unknown;
    /**
     * Information about the model being called.
     */
    readonly modelInfo: CompletionWithToolsModel;
    /**
     * The current {@link GetCompletionWithToolsOptions}
     */
    readonly getCompletionWithToolsOptions: GetCompletionWithToolsOptions | undefined;
};

declare type OnRequestErrorResult = {
    /**
     * If the processor does something to handle the error and wishes for there to be a retry
     * it should set this to how many milliseconds to wait before retrying.
     */
    retryAfter: number;
};

export declare type PendingMessagesModifiedEvent = z.infer<typeof PendingMessagesModifiedEventSchema>;

/**
 * User message queued (ephemeral event for UI updates)
 */
declare const PendingMessagesModifiedEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"pending_messages.modified">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{}, "strip", z.ZodTypeAny, {}, {}>;
}, "strip", z.ZodTypeAny, {
    data: {};
    id: string;
    ephemeral: true;
    type: "pending_messages.modified";
    timestamp: string;
    parentId: string | null;
}, {
    data: {};
    id: string;
    ephemeral: true;
    type: "pending_messages.modified";
    timestamp: string;
    parentId: string | null;
}>;

/**
 * Represents a pending queued item for UI display purposes.
 */
export declare interface PendingQueuedItem {
    kind: "message" | "command";
    displayText: string;
}

/**
 * A permission request which will be used to check tool or path usage against config and/or request user approval.
 */
declare type PermissionRequest = {
    toolCallId?: string;
} & (ShellPermissionRequest | WritePermissionRequest | MCPPermissionRequest | ReadPermissionRequest | UrlPermissionRequest | MemoryPermissionRequest);

/**
 * The result of requesting permissions.
 */
declare type PermissionRequestResult = {
    readonly kind: "approved";
} | {
    readonly kind: "denied-by-rules";
    rules: ReadonlyArray<Rule>;
} | {
    readonly kind: "denied-no-approval-rule-and-could-not-request-from-user";
} | {
    readonly kind: "denied-interactively-by-user";
    readonly feedback?: string;
};

/**
 * Configuration for permissions handling.
 *
 * For CCA, permissions requests are not required.
 */
declare type PermissionsConfig = {
    requestRequired: false;
} | {
    requestRequired: true;
    request: RequestPermissionFn;
};

/**
 * This is just a type to warn that there's a good chance it's not a real path, because
 * it was _very_ heuristically parsed out of a command.
 */
declare type PossiblePath = string;

/**
 * A possible URL that was heuristically parsed from a command.
 */
declare type PossibleUrl = {
    readonly url: string;
};

declare type PostToolExecutionContext = {
    /**
     * The current turn.
     */
    readonly turn: number;
    /**
     * What tool call was executed.
     */
    readonly toolCall: CopilotChatCompletionMessageToolCall;
    /**
     * The result of the tool call. Can be modified in place by the processor.
     */
    readonly toolResult: ToolResultExpanded;
    /**
     * Information about the model being called.
     */
    readonly modelInfo: CompletionWithToolsModel;
};

export declare type PostToolUseHook = (input: PostToolUseHookInput) => Promise<PostToolUseHookOutput | void>;

/**
 * Post-tool use hook types
 */
export declare interface PostToolUseHookInput extends BaseHookInput {
    toolName: string;
    toolArgs: unknown;
    toolResult: ToolResultExpanded;
}

export declare interface PostToolUseHookOutput {
    modifiedResult?: ToolResultExpanded;
    additionalContext?: string;
    suppressOutput?: boolean;
}

declare type PreRequestContext = {
    /**
     * An optional identifier for the completion with tools call. This can be used for logging
     * and tracing purposes.
     */
    readonly callId?: string;
    /**
     * The current turn.
     */
    readonly turn: number;
    /**
     * The current retry attempt.
     */
    readonly retry: number;
    /**
     * The messages that will be sent to the model for the request.
     * If modifying them, do so in place.
     */
    readonly messages: ChatCompletionMessageParam[];
    /**
     * The tool definitions that will be sent to the model for the request.
     * These should not be modified.
     */
    readonly toolDefinitions: ChatCompletionTool[];
    /**
     * Information about the model being called.
     */
    readonly modelInfo: CompletionWithToolsModel;
    /**
     * Additional headers to send with the request. If adding additional headers
     * then simply add to this object.
     */
    readonly additionalRequestHeaders: Record<string, string>;
    /**
     * The current {@link GetCompletionWithToolsOptions}
     */
    readonly getCompletionWithToolsOptions: GetCompletionWithToolsOptions | undefined;
    /**
     * The client that can be used to make additional LLM calls (e.g., for compaction).
     */
    readonly client: Client_2;
    /**
     * The rich tool objects with callbacks, needed for nested LLM calls.
     */
    readonly tools: Tool_2[];
};

declare type PreToolsExecutionContext = {
    /**
     * The current turn.
     */
    readonly turn: number;
    /**
     * What tool calls are being executed.
     */
    readonly toolCalls: CopilotChatCompletionMessageToolCall[];
    /**
     * Information about the model being called.
     */
    readonly modelInfo: CompletionWithToolsModel;
};

/**
 * Pre-tools execution procesors can either return nothing, or results for one or more
 * of the tools to be executed. These results will be given to the model in lieu of
 * performing the tool call and obtaining a result from the tool itself.
 */
declare type PreToolsExecutionResult = void | Map<string, ToolResultExpanded>;

export declare type PreToolUseHook = (input: PreToolUseHookInput) => Promise<PreToolUseHookOutput | void>;

/**
 * Pre-tool use hook types
 */
export declare interface PreToolUseHookInput extends BaseHookInput {
    toolName: string;
    toolArgs: unknown;
}

export declare interface PreToolUseHookOutput {
    permissionDecision?: "allow" | "deny" | "ask";
    permissionDecisionReason?: string;
    modifiedArgs?: unknown;
    additionalContext?: string;
    suppressOutput?: boolean;
}

/**
 * Configuration for a custom API provider (BYOK - Bring Your Own Key).
 * When set, bypasses Copilot API authentication and uses this provider instead.
 */
export declare interface ProviderConfig {
    /** Provider type. Defaults to "openai" for generic OpenAI-compatible APIs. */
    type?: "openai" | "azure" | "anthropic";
    /** Wire API format (openai/azure only). Defaults to "completions". */
    wireApi?: "completions" | "responses";
    /** API endpoint URL */
    baseUrl: string;
    /** API key. Optional for local providers like Ollama. */
    apiKey?: string;
    /**
     * Bearer token for authentication. Sets the Authorization header directly.
     * Use this for services requiring bearer token auth instead of API key.
     * Takes precedence over apiKey when both are set.
     */
    bearerToken?: string;
    /** Azure-specific options */
    azure?: {
        /** API version. Defaults to "2024-10-21". */
        apiVersion?: string;
    };
}

/**
 * Functional query API that provides a Claude Code-inspired interface.
 * This is a thin wrapper around the Session class for simple use cases.
 *
 * @param options - Query configuration options
 * @returns AsyncGenerator of SessionEvent objects
 */
export declare function query(options: QueryOptions): AsyncIterable<SessionEvent>;

/**
 * Hook system with arrays of specific hook callbacks
 */
export declare interface QueryHooks {
    preToolUse?: PreToolUseHook[];
    postToolUse?: PostToolUseHook[];
    userPromptSubmitted?: UserPromptSubmittedHook[];
    sessionStart?: SessionStartHook[];
    sessionEnd?: SessionEndHook[];
    errorOccurred?: ErrorOccurredHook[];
}

export declare type QueryOptions = SessionOptions & {
    prompt: string;
    abortController?: AbortController;
};

/**
 * Callback type for handling queued slash commands.
 * Returns whether the command was handled and whether to stop processing the queue.
 */
export declare type QueuedCommandHandler = (command: string) => Promise<QueuedCommandResult>;

/**
 * Represents a queued slash command to be executed.
 */
export declare interface QueuedCommandItem {
    kind: "command";
    /** The full command string including the slash, e.g., "/compact" or "/model gpt-4" */
    command: string;
}

/**
 * Result from executing a queued slash command.
 * - `stopProcessingQueue`: If true, stops processing remaining items in this session's queue
 *   (used when /clear or /exit wipes the queue)
 */
export declare type QueuedCommandResult = {
    handled: true;
    stopProcessingQueue?: boolean;
} | {
    handled: false;
};

/**
 * A unified queue item that can represent either a user message or a slash command.
 * Used to support queueing slash commands alongside messages in FIFO order.
 */
export declare type QueuedItem = QueuedMessageItem | QueuedCommandItem;

/**
 * Represents a queued user message to be processed by the agentic loop.
 */
export declare interface QueuedMessageItem {
    kind: "message";
    options: SendOptions;
}

declare type QuotaSnapshot = {
    /**
     * Whether or not it's an unlimited entitlement.
     */
    isUnlimitedEntitlement: boolean;
    /**
     * The number of requests included in the entitlement, or "-1" for unlimited
     * entitlement, so that the user/client can understand how much they get each
     * month/period; the value is an integer
     */
    entitlementRequests: number;
    /**
     * The count of requests used so far in this month/period, so that the
     * user/client can understand how much of their entitlement they have used;
     * the value is an integer
     */
    usedRequests: number;
    /**
     * Indicates whether usage is allowed once quota is exhausted, so that the
     * user/client can understand if they can continue usage at a pay-per-request
     * rate when entitlement is exhausted; the value is boolean
     */
    usageAllowedWithExhaustedQuota: boolean;
    /**
     * The count of overage requests made so far in this month/period, so that
     * the user/client can understand how much they have spent in pay-per-request
     * charges so far this month/period; the value is a decimal
     */
    overage: number;
    /**
     * Indicates whether overage is allowed once quota is exhausted, so that the
     * user/client can understand if they can continue usage at a pay-per-request
     * rate when entitlement is exhausted; the value is boolean
     */
    overageAllowedWithExhaustedQuota: boolean;
    /**
     * The percentage of the entitlement remaining at the snapshot timestamp, so
     * that the user/client can understand how much they have remaining and their
     * rate of usage; the value is a decimal
     */
    remainingPercentage: number;
    /**
     * The date when the quota resets, so that the user/client can know when they
     * next receive their entitlement; the value is an RFC3339 formatted UTC date;
     * if the entitlement is unlimited, this value is not included in the snapshot
     */
    resetDate?: Date;
};

/**
 * A permission request for reading file or directory contents.
 */
declare type ReadPermissionRequest = {
    readonly kind: "read";
    /** The intention of the edit operation, e.g. "Read file" or "List directory" */
    readonly intention: string;
    /** The path of the file or directory being read */
    readonly path: string;
};

declare type ReasoningMessageParam = {
    /**
     * An ID or encrypted value that allows the model to restore
     */
    reasoning_opaque?: string;
    /**
     * Human-readable text describing the model's thinking process.
     */
    reasoning_text?: string;
    /**
     * An encrypted representation of the model's internal state
     */
    encrypted_content?: string | undefined | null;
};

/**
 * Human-readable labels for each relevance score level.
 */
declare const RELEVANCE_LABELS: Record<RelevanceScore, string>;

/**
 * Relevance score for a session relative to the current working directory.
 * Higher scores indicate more relevant sessions.
 */
declare type RelevanceScore = 0 | 1 | 2 | 3 | 4;

export declare class RemoteSession extends Session<RemoteSessionMetadata> {
    readonly repository: {
        name: string;
        owner: string;
        branch: string;
    };
    readonly remoteSessionIds: string[];
    readonly pullRequestNumber?: number;
    constructor(options: SessionOptions & {
        repository: {
            name: string;
            owner: string;
            branch: string;
        };
        remoteSessionIds: string[];
        pullRequestNumber?: number;
    });
    send(options: SendOptions): Promise<void>;
    abort(): Promise<void>;
    compactHistory(): Promise<CompactionResult>;
    getMetadata(): RemoteSessionMetadata;
}

export declare interface RemoteSessionMetadata extends SessionMetadata {
    readonly repository: {
        owner: string;
        name: string;
        branch: string;
    };
    readonly remoteSessionIds: string[];
    readonly pullRequestNumber?: number;
    readonly isRemote: true;
}

declare type RequestPermissionFn = (permission: PermissionRequest) => Promise<PermissionRequestResult>;

/**
 * An event that is emitted by the `Client` which contains the final response from the LLM.
 */
declare type ResponseEvent = {
    kind: "response";
    turn?: number;
    callId?: string;
    modelCall?: ModelCallParam;
    response: ChatCompletionMessage;
};

/**
 * Retrieves the list of available models based on policies and integration including capabilities and
 * billing information, which may be cached from previous calls.
 *
 * This list is in order of preference to be the default model for new sessions where the first model is
 * the most preferred.  It can be empty if all model policies are disabled.
 */
export declare function retrieveAvailableModels(authInfo: AuthInfo, copilotUrl: string | undefined, integrationId: string, sessionId: string, logger: RunnerLogger): Promise<{
    models: Model[];
    copilotUrl: string | undefined;
}>;

/**
 * A Rule defines a pattern for matching permission requests.
 *
 * It is unfortunately generically named because it is intended to match across
 * different types of tool uses, e.g. `Shell(touch)` or `GitHubMCP(list_issues)`,
 * `view(.env-secrets)`
 */
declare type Rule = {
    /**
     * The kind of rule that should be matched e.g. `Shell` or `GitHubMCP`.
     */
    readonly kind: string;
    /**
     * If null, matches all arguments to the kind.
     */
    readonly argument: string | null;
};

export declare interface RunnerLogger {
    /**
     * Log a message ignoring the configured log level.
     * This is useful for logging messages that should always be logged, regardless of the log level.
     * @param message The message to log.
     */
    log(message: string): void;
    /**
     * Returns true if the environment is set to debug.
     * Note: This is not the same as the log level being set to debug.
     */
    isDebug(): boolean;
    /**
     * Log a debug message. This is only logged if the log level is set to debug.
     * @param message The message to log.
     */
    debug(message: string): void;
    /**
     * Log an info message. This is only logged if the log level is set to info or debug.
     * @param message The message to log.
     */
    info(message: string): void;
    /**
     * Log a notice message. This is only logged if the log level is set to warning, info, or debug,
     * but logs using the logger's info method.
     * This is useful for logging messages that are not errors, but are important enough to log on
     * less verbose log levels.
     * @param message The message to log.
     */
    notice(message: string | Error): void;
    /**
     * Log a warning message. This is only logged if the log level is set to warning, info, or debug
     * @param message The message to log.
     */
    warning(message: string | Error): void;
    /**
     * Log an error message. This is only logged if the log level is set to error, warning, info, or debug
     * @param message The message to log.
     */
    error(message: string | Error): void;
    /**
     * Log a message that starts a new group.
     * @param name The name of the group.
     * @param level The log level of the group. Defaults to info.
     */
    startGroup(name: string, level?: LogLevel): void;
    /**
     * Log a message that ends the current group.
     * @param level The log level of the group. Defaults to info.
     */
    endGroup(level?: LogLevel): void;
}

declare type RuntimeSettings = DeepPartial<IRuntimeSettings>;

/**
 * Indicates the safety level of the assessed script.
 *
 * There may some cases where a script cannot be assessed (e.g. unparseable),
 * in which case we bail with a failure.
 */
declare type SafetyAssessment = {
    readonly result: "failed";
    /**
     * A human-readable reason why the safety assessment could not be completed.
     */
    readonly reason: string;
} | {
    readonly result: "completed";
    readonly commands: ReadonlyArray<AssessedCommand>;
    /**
     * Possible absolute file paths that the script might operate on, based on heuristic parsing.
     */
    readonly possiblePaths: ReadonlyArray<PossiblePath>;
    /**
     * Possible URLs that the script might access, based on heuristic parsing.
     */
    readonly possibleUrls: ReadonlyArray<PossibleUrl>;
    /**
     * Indicates whether any command in the script has redirection to write to a file.
     */
    readonly hasWriteFileRedirection: boolean;
    /**
     * Indicates whether a command can be approved for the rest of the running session.
     *
     * Simple commands like `git status` or `npm test` can be session approved
     * because their impact is predictable. Complex commands with substitutions,
     * variables, or side effects require per-invocation approval.
     *
     * Examples of session-approvable: `ls`, `git log`, `command1 && command2`
     * Examples requiring per-invocation: `rm -rf $DIR`, `find -exec`
     */
    readonly canOfferSessionApproval: boolean;
    /**
     * Optional warning message to display to the user (e.g., when parser is unavailable).
     * This should be shown in the permission dialog and/or timeline.
     */
    readonly warning?: string;
};

/**
 * Session metadata with its relevance score for display grouping.
 */
declare interface ScoredSessionMetadata<T extends SessionMetadata = SessionMetadata> {
    session: T;
    score: RelevanceScore;
}

/**
 * Scores the relevance of a session's context to the current working directory context.
 * Higher scores indicate more relevant sessions.
 *
 * Scoring:
 * - 4: Same repository + same branch (exact match)
 * - 3: Same repository, different branch
 * - 2: Same git root (for non-GitHub repos)
 * - 1: Same working directory (for non-git directories)
 * - 0: No match
 */
declare function scoreSessionRelevance(sessionContext: SessionContext | undefined, currentContext: WorkingDirectoryContext): RelevanceScore;

export declare type SelectionAttachment = z.infer<typeof SelectionAttachmentSchema>;

declare const SelectionAttachmentSchema: z.ZodObject<{
    type: z.ZodLiteral<"selection">;
    filePath: z.ZodString;
    displayName: z.ZodString;
    text: z.ZodString;
    selection: z.ZodObject<{
        start: z.ZodObject<{
            line: z.ZodNumber;
            character: z.ZodNumber;
        }, "strip", z.ZodTypeAny, {
            line: number;
            character: number;
        }, {
            line: number;
            character: number;
        }>;
        end: z.ZodObject<{
            line: z.ZodNumber;
            character: z.ZodNumber;
        }, "strip", z.ZodTypeAny, {
            line: number;
            character: number;
        }, {
            line: number;
            character: number;
        }>;
    }, "strip", z.ZodTypeAny, {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    }, {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    }>;
}, "strip", z.ZodTypeAny, {
    type: "selection";
    text: string;
    filePath: string;
    displayName: string;
    selection: {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    };
}, {
    type: "selection";
    text: string;
    filePath: string;
    displayName: string;
    selection: {
        end: {
            line: number;
            character: number;
        };
        start: {
            line: number;
            character: number;
        };
    };
}>;

export declare interface SendOptions {
    prompt: string;
    /** If provided, this is shown in the timeline instead of prompt */
    displayPrompt?: string;
    attachments?: Attachment[];
    mode?: "enqueue" | "immediate";
    abortController?: AbortController;
    /**
     * If true, adds the message to the front of the queue instead of the end.
     * This is useful when a queued command (e.g., /plan, /review) returns an agentMessage
     * that should be processed immediately after the command, before other queued items.
     * Without this, the agentMessage would be appended to the end and processed after
     * any other items already in the queue.
     */
    prepend?: boolean;
}

export declare abstract class Session<SM extends SessionMetadata = SessionMetadata> {
    readonly sessionId: string;
    readonly startTime: Date;
    readonly modifiedTime: Date;
    readonly summary?: string;
    private events;
    protected _chatMessages: ChatCompletionMessageParam[];
    protected _selectedModel: string | undefined;
    protected originalUserMessages: string[];
    private compactionCheckpointLength;
    private eventProcessingQueue;
    private eventHandlers;
    private wildcardEventHandlers;
    protected integrationId: string;
    protected availableTools?: string[];
    protected excludedTools?: string[];
    protected executeToolsInParallel?: boolean;
    protected shellConfig?: ShellConfig;
    protected requestPermission?: (permissionRequest: PermissionRequest) => Promise<PermissionRequestResult>;
    protected mcpServers?: Record<string, MCPServerConfig>;
    protected mcpHost?: McpHost;
    protected selectedAgentMcpServerNames: Set<string>;
    protected lastInitializedAgentMcpServers?: Record<string, MCPServerConfig>;
    protected hooks?: QueryHooks;
    protected customAgents?: SweCustomAgent[];
    protected selectedCustomAgent?: SweCustomAgent;
    protected organizationCustomInstructions?: string;
    protected skipCustomInstructions: boolean;
    protected systemMessageConfig?: SystemMessageConfig;
    protected workingDir: string;
    protected featureFlags?: FeatureFlags;
    protected skillDirectories?: string[];
    protected disabledSkills?: Set<string>;
    protected installedPlugins?: InstalledPlugin[];
    protected onFileCreated?: (path: string) => void;
    protected onWarning?: (message: string) => void;
    protected requestUserInput?: (request: {
        question: string;
        choices?: string[];
        allowFreeform?: boolean;
    }) => Promise<{
        answer: string;
        wasFreeform: boolean;
    }>;
    protected askUserDisabled?: boolean;
    protected runningInInteractiveMode?: boolean;
    protected trajectoryFile?: string;
    protected eventsLogDirectory?: string;
    protected authInfo?: AuthInfo;
    protected copilotUrl?: string;
    protected provider?: ProviderConfig;
    protected onToolsUpdate?: (tools: Tool_2[]) => void;
    protected enableStreaming: boolean;
    protected handoffContext?: string;
    protected externalToolDefinitions?: ExternalToolDefinition[];
    protected externalToolDispatcher?: ExternalToolDispatcher;
    protected clientName?: string;
    protected largeOutputConfig?: LargeToolOutputConfig;
    protected runtimeSettings?: RuntimeSettings;
    protected configDir?: string;
    protected repositoryName?: string;
    protected workspaceContext?: WorkspaceContextInfo;
    protected currentSystemMessage?: string;
    protected currentToolMetadata?: ToolMetadata[];
    protected currentTools?: Tool_2[];
    abstract send(options: SendOptions): Promise<void>;
    abstract abort(): Promise<void>;
    /**
     * Check if the session is currently in a state where it can be aborted.
     * Returns true if there's an active operation that can be cancelled.
     * Default implementation returns false. Override in subclasses that support abortion.
     */
    isAbortable(): boolean;
    /**
     * Initializes tools and validates tool filter configuration.
     * This method should be called after the session is fully configured (auth, model, MCP servers)
     * but before the first message is sent. It will emit warnings for any unknown tool names
     * specified in availableTools or excludedTools.
     *
     * Default implementation is a no-op. Override in subclasses that support tool validation.
     *
     * @returns Promise that resolves when initialization and validation is complete
     */
    initializeAndValidateTools(): Promise<void>;
    constructor(options?: SessionOptions);
    abstract getMetadata(): SM;
    static fromEvents<S extends Session, O extends SessionOptions = SessionOptions>(this: new (_options: O) => S, events: SessionEvent[], options?: O): Promise<S>;
    /**
     * Updates the authentication information for this session.
     * This is a convenience method equivalent to updateOptions({ authInfo }).
     *
     * @param authInfo - The new authentication information to use for this session
     */
    setAuthInfo(authInfo: AuthInfo | undefined): void;
    /**
     * Updates session options after creation.
     * This method allows selectively updating configuration options without recreating the session.
     * Only the provided options will be updated; omitted options remain unchanged.
     *
     * @param options - Partial session options to update
     *
     * @example
     * ```typescript
     * // Update multiple options at once
     * session.updateOptions({
     *   logger: fileLogger,
     *   mcpServers: mcpConfig,
     *   customAgents: loadedAgents
     * });
     *
     * // Or use convenience methods for single updates
     * session.setAuthInfo(newAuthInfo); // Preferred for auth
     * session.setSelectedModel(newModel); // Preferred for model
     * ```
     */
    updateOptions(options: Partial<UpdatableSessionOptions>): void;
    /**
     * Set a callback to be notified when tools are initialized/updated.
     * This is called during the agentic loop after tools are built and available.
     * Useful for UI components that need to display tool information.
     *
     * @param callback - Function to call with the available tools
     *
     * @example
     * ```typescript
     * session.setOnToolsUpdate((tools) => {
     *   console.log(`${tools.length} tools available`);
     *   updateToolsList(tools);
     * });
     * ```
     */
    setOnToolsUpdate(callback: (tools: Tool_2[]) => void): void;
    /**
     * Get list of available custom agents.
     * Returns the custom agents that were provided when the session was created.
     *
     * @returns Array of available custom agents
     */
    getAvailableCustomAgents(): SweCustomAgent[];
    /**
     * Get the custom configuration directory for this session.
     * When set, this overrides the default Copilot config directory (~/.copilot or $XDG_CONFIG_HOME/.copilot).
     *
     * @returns The custom configuration directory, or undefined if using the default
     */
    getConfigDir(): string | undefined;
    protected getRuntimeSettings(): RuntimeSettings;
    /**
     * Resolves the repository name to use for settings.
     * Returns the explicitly configured repositoryName if set,
     * otherwise attempts to detect it from the git remote in the working directory.
     *
     * @returns The repository name in "owner/repo" format, or undefined if not available
     */
    protected resolveRepositoryName(): Promise<string | undefined>;
    /**
     * Select a custom agent for subsequent queries.
     * When a custom agent is selected, only that agent's tools (and required tools) will be available.
     * The selected custom agent will also be available as a tool to the main agent.
     *
     * @param agentId - The name/id of the custom agent to select
     * @throws Error if the agent is not found
     */
    selectCustomAgent(agentId: string): Promise<void>;
    /**
     * Clear custom agent selection.
     * After calling this, all tools will be available again and all custom agents
     * will be available as tools.
     */
    clearCustomAgent(): void;
    /**
     * Registers an event handler for specific event types or all events.
     * Supports both synchronous and asynchronous handlers.
     *
     * @param eventType - The event type to listen for (e.g., "assistant.message", "tool.execution_complete") or "*" for all events
     * @param handler - The handler function to call when the event is emitted. Can be sync or async.
     * @returns A function that unsubscribes the handler when called
     */
    on<K extends EventType>(eventType: K, handler: EventHandler<K>): () => void;
    on(eventType: "*", handler: WildcardEventHandler): () => void;
    /**
     * Emits an event to all registered handlers.
     * Automatically generates event fields (id, timestamp, parentId) and adds the event to the session history.
     * Triggers both legacy callbacks and new-style event handlers.
     *
     * @param eventType - The type of event to emit
     * @param data - The event data payload
     * @param ephemeral - Whether this event should be persisted (default: false)
     */
    private emitInternal;
    /**
     * Emits an event to all registered handlers.
     * Automatically generates event fields (id, timestamp, parentId) and adds the event to the session history.
     * Triggers both legacy callbacks and new-style event handlers. Does not allow emitting ephemeral events.
     *
     * @param eventType - The type of event to emit
     * @param data - The event data payload
     */
    emit<K extends EventType>(eventType: EventPayload<K>["ephemeral"] extends true ? never : K, data: EventData<K>): void;
    /**
     * Emits an ephemeral event (not persisted to disk).
     * Convenience method that calls emit() with ephemeral = true.
     *
     * @param eventType - The type of event to emit
     * @param data - The event data payload
     */
    emitEphemeral<K extends EventType>(eventType: EventPayload<K>["ephemeral"] extends false ? never : K, data: EventData<K>): void;
    /**
     * Returns all events that have occurred in the session.
     * Events are returned in chronological order and include all session lifecycle, user messages,
     * assistant responses, tool executions, and other session events.
     *
     * @returns A readonly array of all session events
     */
    getEvents(): readonly SessionEvent[];
    /**
     * Truncates the session to a specific event, removing all events after it.
     * This clears and rebuilds the internal state (chat messages, etc.) from the remaining events.
     *
     * Note: This only affects in-memory state. The caller is responsible for
     * persisting the truncation to disk (e.g., via SessionEventState.truncate).
     *
     * @param upToEventId - The event ID to truncate to (this event is excluded)
     * @returns The number of events removed
     */
    truncateToEvent(upToEventId: string): Promise<{
        eventsRemoved: number;
    }>;
    /**
     * Returns all chat messages reconstructed from session events.
     * Messages are processed asynchronously in order to handle attachments and ensure consistency.
     * Includes user messages, assistant messages, tool messages, and system messages.
     *
     * @returns A Promise that resolves to a readonly array of chat completion messages
     * ```
     */
    getChatMessages(): Promise<readonly ChatCompletionMessageParam[]>;
    /**
     * Returns chat messages that are part of the conversation context (excludes system messages).
     * Useful for displaying the user/assistant conversation without system prompts.
     *
     * @returns A Promise that resolves to an array of user, assistant, and tool messages
     */
    getChatContextMessages(): Promise<ChatCompletionMessageParam[]>;
    /**
     * Returns only system and developer messages from the chat history.
     * Useful for inspecting the system prompts and instructions given to the model.
     *
     * @returns A Promise that resolves to an array of system/developer messages
     */
    getSystemContextMessages(): Promise<ChatCompletionMessageParam[]>;
    /**
     * Returns the current system message used in the most recent agent turn.
     * This is the full system prompt that was sent to the model.
     *
     * @returns The system message string, or undefined if not yet initialized
     */
    getCurrentSystemMessage(): string | undefined;
    /**
     * Returns lightweight tool metadata for token counting.
     * Used by /context command to calculate token usage.
     *
     * @returns The array of tool metadata, or undefined if not yet initialized
     */
    getCurrentToolMetadata(): ToolMetadata[] | undefined;
    /**
     * Returns the currently selected model for this session.
     * The model may change during a session if `setSelectedModel` is called.
     *
     * @returns A Promise that resolves to the model identifier string, or undefined if no model is set
     */
    getSelectedModel(): Promise<string | undefined>;
    /**
     * Returns the organization custom instructions configured for this session.
     * These are additional instructions provided by the organization.
     *
     * @returns The organization custom instructions string, or undefined if not set
     */
    getOrganizationCustomInstructions(): string | undefined;
    /**
     * Returns whether custom instructions should be skipped for this session.
     *
     * @returns True if custom instructions should be skipped, false otherwise
     */
    getSkipCustomInstructions(): boolean;
    /**
     * Returns the system message configuration for this session.
     * This can be used to replace or append to the default system message.
     *
     * @returns The system message config, or undefined if not set
     */
    getSystemMessageConfig(): SystemMessageConfig | undefined;
    /**
     * Sets the workspace context for infinite sessions.
     * This context is injected into the system prompt to give the agent
     * awareness of the workspace and its history.
     *
     * @param context - The workspace context info, or undefined to clear
     */
    setWorkspaceContext(context: WorkspaceContextInfo | undefined): void;
    /**
     * Gets the current workspace context, if any.
     */
    getWorkspaceContext(): WorkspaceContextInfo | undefined;
    /**
     * Get the current workspace, if any.
     * Only available for LocalSession when infinite sessions are enabled.
     */
    getWorkspace(): Workspace | null;
    /**
     * Check if workspace features are enabled for this session.
     */
    isWorkspaceEnabled(): boolean;
    /**
     * Get the workspace path for this session.
     * Returns null for base Session (non-local sessions don't have workspaces).
     */
    getWorkspacePath(): string | null;
    /**
     * Get the number of checkpoints in the workspace.
     */
    getCheckpointCount(): number;
    /**
     * List checkpoints with their titles for context injection.
     */
    listCheckpointTitles(): Promise<{
        number: number;
        title: string;
        filename: string;
    }[]>;
    /**
     * Check if a plan.md file exists in the workspace.
     */
    hasPlan(): boolean;
    /**
     * Ensure workspace exists for this session.
     * Only available for LocalSession when infinite sessions are enabled.
     */
    ensureWorkspace(_context?: WorkspaceContext): Promise<Workspace>;
    /**
     * Sets the selected model for this session and emits a model change event.
     * This allows switching models mid-session, with the change tracked in the event history.
     *
     * @param model - The model identifier to switch to
     * @returns A Promise that resolves when the model change event has been emitted
     */
    setSelectedModel(model: string): Promise<void>;
    /**
     * Compacts the conversation history into a single summary message.
     * Used by the /compact slash command for manual compaction.
     *
     * @returns Promise that resolves with compaction results
     * @throws Error if compaction fails or is not supported
     */
    abstract compactHistory(): Promise<CompactionResult>;
    /**
     * Get the ID of the last event (for parentId chaining)
     */
    private getLastEventId;
    /**
     * Add a function to the event processing queue to ensure sequential processing
     * Returns a promise that resolves when the function has been processed
     * Ensures that state updates from events are processed in order
     */
    protected enqueueEventProcessing<T>(fn: () => T | PromiseLike<T>): Promise<T>;
    /**
     * Process event to update internal state (_chatMessages, _selectedModel, etc.)
     */
    private processEventForState;
    private loadCustomAgents;
    /**
     * Applies compaction to chat messages by splitting at the checkpoint and creating
     * post-compaction messages. Updates _chatMessages and returns the compacted and new messages.
     *
     * @param checkpointLength - The number of messages at the compaction checkpoint
     * @param summaryContent - The summary content from compaction
     * @returns The compacted messages and any new messages added after the checkpoint
     */
    protected applyCompactionToMessages(checkpointLength: number, summaryContent: string): {
        compacted: ChatCompletionMessageParam[];
        newMessages: ChatCompletionMessageParam[];
    };
}

export declare type SessionCompactionCompleteData = SessionCompactionCompleteEvent["data"];

export declare type SessionCompactionCompleteEvent = z.infer<typeof SessionCompactionCompleteEventSchema>;

/**
 * Session compaction complete event - conversation history compaction finished (success or failure)
 */
declare const SessionCompactionCompleteEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.compaction_complete">;
    data: z.ZodObject<{
        success: z.ZodBoolean;
        error: z.ZodOptional<z.ZodString>;
        preCompactionTokens: z.ZodOptional<z.ZodNumber>;
        postCompactionTokens: z.ZodOptional<z.ZodNumber>;
        preCompactionMessagesLength: z.ZodOptional<z.ZodNumber>;
        messagesRemoved: z.ZodOptional<z.ZodNumber>;
        tokensRemoved: z.ZodOptional<z.ZodNumber>;
        summaryContent: z.ZodOptional<z.ZodString>;
        compactionTokensUsed: z.ZodOptional<z.ZodObject<{
            input: z.ZodNumber;
            output: z.ZodNumber;
            cachedInput: z.ZodNumber;
        }, "strip", z.ZodTypeAny, {
            input: number;
            output: number;
            cachedInput: number;
        }, {
            input: number;
            output: number;
            cachedInput: number;
        }>>;
    }, "strip", z.ZodTypeAny, {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    }, {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    };
    id: string;
    type: "session.compaction_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    };
    id: string;
    type: "session.compaction_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SessionCompactionStartEvent = z.infer<typeof SessionCompactionStartEventSchema>;

/**
 * Session compaction started event - conversation history compaction has begun
 */
declare const SessionCompactionStartEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.compaction_start">;
    data: z.ZodObject<{}, "strip", z.ZodTypeAny, {}, {}>;
}, "strip", z.ZodTypeAny, {
    data: {};
    id: string;
    type: "session.compaction_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {};
    id: string;
    type: "session.compaction_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

/**
 * Working directory context for session tracking
 */
export declare interface SessionContext {
    readonly cwd: string;
    readonly gitRoot?: string;
    readonly repository?: string;
    readonly branch?: string;
}

export declare type SessionEndHook = (input: SessionEndHookInput) => Promise<SessionEndHookOutput | void>;

/**
 * Session end hook types
 */
export declare interface SessionEndHookInput extends BaseHookInput {
    reason: "complete" | "error" | "abort" | "timeout" | "user_exit";
    finalMessage?: string;
    error?: Error;
}

export declare interface SessionEndHookOutput {
    suppressOutput?: boolean;
    cleanupActions?: string[];
    sessionSummary?: string;
}

export declare type SessionErrorEvent = z.infer<typeof SessionErrorEventSchema>;

/**
 * Error notification (for timeline/UI display)
 */
declare const SessionErrorEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.error">;
    data: z.ZodObject<{
        errorType: z.ZodString;
        message: z.ZodString;
        stack: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        message: string;
        errorType: string;
        stack?: string | undefined;
    }, {
        message: string;
        errorType: string;
        stack?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        message: string;
        errorType: string;
        stack?: string | undefined;
    };
    id: string;
    type: "session.error";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        message: string;
        errorType: string;
        stack?: string | undefined;
    };
    id: string;
    type: "session.error";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

/**
 * Union of all session event types
 */
export declare type SessionEvent = z.infer<typeof SessionEventSchema>;

/**
 * Discriminated union of all event schemas
 */
export declare const SessionEventSchema: z.ZodDiscriminatedUnion<"type", [z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.start">;
    data: z.ZodObject<{
        sessionId: z.ZodString;
        version: z.ZodNumber;
        producer: z.ZodString;
        copilotVersion: z.ZodString;
        startTime: z.ZodString;
        selectedModel: z.ZodOptional<z.ZodString>;
        context: z.ZodOptional<z.ZodObject<{
            cwd: z.ZodString;
            gitRoot: z.ZodOptional<z.ZodString>;
            repository: z.ZodOptional<z.ZodString>;
            branch: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    }, {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    };
    id: string;
    type: "session.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    };
    id: string;
    type: "session.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.resume">;
    data: z.ZodObject<{
        resumeTime: z.ZodString;
        eventCount: z.ZodNumber;
        context: z.ZodOptional<z.ZodObject<{
            cwd: z.ZodString;
            gitRoot: z.ZodOptional<z.ZodString>;
            repository: z.ZodOptional<z.ZodString>;
            branch: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    }, {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    };
    id: string;
    type: "session.resume";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    };
    id: string;
    type: "session.resume";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.error">;
    data: z.ZodObject<{
        errorType: z.ZodString;
        message: z.ZodString;
        stack: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        message: string;
        errorType: string;
        stack?: string | undefined;
    }, {
        message: string;
        errorType: string;
        stack?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        message: string;
        errorType: string;
        stack?: string | undefined;
    };
    id: string;
    type: "session.error";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        message: string;
        errorType: string;
        stack?: string | undefined;
    };
    id: string;
    type: "session.error";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"session.idle">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{}, "strip", z.ZodTypeAny, {}, {}>;
}, "strip", z.ZodTypeAny, {
    data: {};
    id: string;
    ephemeral: true;
    type: "session.idle";
    timestamp: string;
    parentId: string | null;
}, {
    data: {};
    id: string;
    ephemeral: true;
    type: "session.idle";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.info">;
    data: z.ZodObject<{
        infoType: z.ZodString;
        message: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        message: string;
        infoType: string;
    }, {
        message: string;
        infoType: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        message: string;
        infoType: string;
    };
    id: string;
    type: "session.info";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        message: string;
        infoType: string;
    };
    id: string;
    type: "session.info";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.model_change">;
    data: z.ZodObject<{
        previousModel: z.ZodOptional<z.ZodString>;
        newModel: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        newModel: string;
        previousModel?: string | undefined;
    }, {
        newModel: string;
        previousModel?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        newModel: string;
        previousModel?: string | undefined;
    };
    id: string;
    type: "session.model_change";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        newModel: string;
        previousModel?: string | undefined;
    };
    id: string;
    type: "session.model_change";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.import_legacy">;
    data: z.ZodObject<{
        legacySession: z.ZodObject<{
            sessionId: z.ZodString;
            startTime: z.ZodDate;
            chatMessages: z.ZodArray<z.ZodUnion<[z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, "many">]>;
                role: z.ZodLiteral<"developer">;
                name: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }, {
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }>, z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, "many">]>;
                role: z.ZodLiteral<"system">;
                name: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }, {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }>, z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodUnion<[z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"image_url">;
                    image_url: z.ZodObject<{
                        url: z.ZodString;
                        detail: z.ZodOptional<z.ZodEnum<["auto", "low", "high"]>>;
                    }, "strip", z.ZodTypeAny, {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    }, {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                }, {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"input_audio">;
                    input_audio: z.ZodObject<{
                        data: z.ZodString;
                        format: z.ZodUnion<[z.ZodLiteral<"wav">, z.ZodLiteral<"mp3">]>;
                    }, "strip", z.ZodTypeAny, {
                        data: string;
                        format: "wav" | "mp3";
                    }, {
                        data: string;
                        format: "wav" | "mp3";
                    }>;
                }, "strip", z.ZodTypeAny, {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                }, {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"file">;
                    file: z.ZodObject<{
                        file_date: z.ZodOptional<z.ZodString>;
                        file_id: z.ZodOptional<z.ZodString>;
                        filename: z.ZodOptional<z.ZodString>;
                    }, "strip", z.ZodTypeAny, {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    }, {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                }, {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                }>]>, "many">]>;
                role: z.ZodLiteral<"user">;
                name: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            }, {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            }>, z.ZodObject<{
                content: z.ZodOptional<z.ZodNullable<z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodUnion<[z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"refusal">;
                    refusal: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    refusal: string;
                    type: "refusal";
                }, {
                    refusal: string;
                    type: "refusal";
                }>]>, "many">]>>>;
                role: z.ZodLiteral<"assistant">;
                name: z.ZodOptional<z.ZodString>;
                refusal: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                audio: z.ZodOptional<z.ZodNullable<z.ZodObject<{
                    id: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    id: string;
                }, {
                    id: string;
                }>>>;
                function_call: z.ZodOptional<z.ZodNullable<z.ZodObject<{
                    name: z.ZodString;
                    arguments: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    name: string;
                    arguments: string;
                }, {
                    name: string;
                    arguments: string;
                }>>>;
                tool_calls: z.ZodOptional<z.ZodArray<z.ZodUnion<[z.ZodObject<{
                    id: z.ZodString;
                    type: z.ZodLiteral<"function">;
                    function: z.ZodObject<{
                        name: z.ZodString;
                        arguments: z.ZodString;
                    }, "strip", z.ZodTypeAny, {
                        name: string;
                        arguments: string;
                    }, {
                        name: string;
                        arguments: string;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                }, {
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                }>, z.ZodObject<{
                    id: z.ZodString;
                    type: z.ZodLiteral<"custom">;
                    custom: z.ZodObject<{
                        name: z.ZodString;
                        input: z.ZodString;
                    }, "strip", z.ZodTypeAny, {
                        input: string;
                        name: string;
                    }, {
                        input: string;
                        name: string;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                }, {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                }>]>, "many">>;
            }, "strip", z.ZodTypeAny, {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            }, {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            }>, z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, "many">]>;
                role: z.ZodLiteral<"tool">;
                tool_call_id: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            }, {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            }>, z.ZodObject<{
                content: z.ZodNullable<z.ZodString>;
                role: z.ZodLiteral<"function">;
                name: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                name: string;
                role: "function";
                content: string | null;
            }, {
                name: string;
                role: "function";
                content: string | null;
            }>]>, "many">;
            timeline: z.ZodArray<z.ZodIntersection<z.ZodUnion<[z.ZodObject<{
                type: z.ZodLiteral<"copilot">;
                text: z.ZodString;
                isStreaming: z.ZodOptional<z.ZodBoolean>;
            }, "strip", z.ZodTypeAny, {
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            }, {
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"error">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "error";
                text: string;
            }, {
                type: "error";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"info">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "info";
                text: string;
            }, {
                type: "info";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"warning">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "warning";
                text: string;
            }, {
                type: "warning";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"user">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "user";
                text: string;
            }, {
                type: "user";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"tool_call_requested">;
                callId: z.ZodString;
                name: z.ZodString;
                toolTitle: z.ZodOptional<z.ZodString>;
                intentionSummary: z.ZodNullable<z.ZodString>;
                arguments: z.ZodUnion<[z.ZodUnion<[z.ZodObject<{
                    command: z.ZodString;
                    description: z.ZodString;
                    timeout: z.ZodOptional<z.ZodNumber>;
                    sessionId: z.ZodOptional<z.ZodString>;
                    async: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    input: z.ZodString;
                    delay: z.ZodOptional<z.ZodNumber>;
                }, "strip", z.ZodTypeAny, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    delay: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                    delay: number;
                }, {
                    sessionId: string;
                    delay: number;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                }, {
                    sessionId: string;
                }>]>, z.ZodDiscriminatedUnion<"command", [z.ZodObject<{
                    command: z.ZodLiteral<"view">;
                    path: z.ZodString;
                    view_range: z.ZodOptional<z.ZodTuple<[z.ZodNumber, z.ZodNumber], null>>;
                }, "strip", z.ZodTypeAny, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"create">;
                    path: z.ZodString;
                    file_text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "create";
                    path: string;
                    file_text: string;
                }, {
                    command: "create";
                    path: string;
                    file_text: string;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"str_replace">;
                    path: z.ZodString;
                    new_str: z.ZodOptional<z.ZodString>;
                    old_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"insert">;
                    path: z.ZodString;
                    insert_line: z.ZodNumber;
                    new_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }>]>, z.ZodUnknown]>;
                partialOutput: z.ZodOptional<z.ZodString>;
                isHidden: z.ZodOptional<z.ZodBoolean>;
                isAlwaysExpanded: z.ZodOptional<z.ZodBoolean>;
                showNoContent: z.ZodOptional<z.ZodBoolean>;
            }, "strip", z.ZodTypeAny, {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }, {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"tool_call_completed">;
                callId: z.ZodString;
                name: z.ZodString;
                toolTitle: z.ZodOptional<z.ZodString>;
                intentionSummary: z.ZodNullable<z.ZodString>;
                result: z.ZodUnion<[z.ZodObject<{
                    type: z.ZodLiteral<"success">;
                    log: z.ZodString;
                    detailedLog: z.ZodOptional<z.ZodString>;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                }, {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"failure">;
                    log: z.ZodString;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                }, {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"rejected">;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    type: "rejected";
                    markdown?: boolean | undefined;
                }, {
                    type: "rejected";
                    markdown?: boolean | undefined;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"denied">;
                    log: z.ZodString;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                }, {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                }>]>;
                arguments: z.ZodUnion<[z.ZodUnion<[z.ZodObject<{
                    command: z.ZodString;
                    description: z.ZodString;
                    timeout: z.ZodOptional<z.ZodNumber>;
                    sessionId: z.ZodOptional<z.ZodString>;
                    async: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    input: z.ZodString;
                    delay: z.ZodOptional<z.ZodNumber>;
                }, "strip", z.ZodTypeAny, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    delay: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                    delay: number;
                }, {
                    sessionId: string;
                    delay: number;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                }, {
                    sessionId: string;
                }>]>, z.ZodDiscriminatedUnion<"command", [z.ZodObject<{
                    command: z.ZodLiteral<"view">;
                    path: z.ZodString;
                    view_range: z.ZodOptional<z.ZodTuple<[z.ZodNumber, z.ZodNumber], null>>;
                }, "strip", z.ZodTypeAny, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"create">;
                    path: z.ZodString;
                    file_text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "create";
                    path: string;
                    file_text: string;
                }, {
                    command: "create";
                    path: string;
                    file_text: string;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"str_replace">;
                    path: z.ZodString;
                    new_str: z.ZodOptional<z.ZodString>;
                    old_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"insert">;
                    path: z.ZodString;
                    insert_line: z.ZodNumber;
                    new_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }>]>, z.ZodUnknown]>;
                isHidden: z.ZodOptional<z.ZodBoolean>;
                isAlwaysExpanded: z.ZodOptional<z.ZodBoolean>;
                showNoContent: z.ZodOptional<z.ZodBoolean>;
            }, "strip", z.ZodTypeAny, {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }, {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }>]>, z.ZodObject<{
                id: z.ZodString;
                timestamp: z.ZodDate;
            }, "strip", z.ZodTypeAny, {
                id: string;
                timestamp: Date;
            }, {
                id: string;
                timestamp: Date;
            }>>, "many">;
            selectedModel: z.ZodOptional<z.ZodEnum<["claude-sonnet-4.5", "claude-haiku-4.5", "claude-opus-4.5", "claude-sonnet-4", "gpt-5.2-codex", "gpt-5.1-codex-max", "gpt-5.1-codex", "gpt-5.2", "gpt-5.1", "gpt-5", "gpt-5.1-codex-mini", "gpt-5-mini", "gpt-4.1", "gemini-3-pro-preview"]>>;
        }, "strip", z.ZodTypeAny, {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        }, {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        }>;
        importTime: z.ZodString;
        sourceFile: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    }, {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    };
    id: string;
    type: "session.import_legacy";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    };
    id: string;
    type: "session.import_legacy";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.handoff">;
    data: z.ZodObject<{
        handoffTime: z.ZodString;
        sourceType: z.ZodEnum<["remote", "local"]>;
        repository: z.ZodOptional<z.ZodObject<{
            owner: z.ZodString;
            name: z.ZodString;
            branch: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            owner: string;
            name: string;
            branch?: string | undefined;
        }, {
            owner: string;
            name: string;
            branch?: string | undefined;
        }>>;
        context: z.ZodOptional<z.ZodString>;
        summary: z.ZodOptional<z.ZodString>;
        remoteSessionId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    }, {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    };
    id: string;
    type: "session.handoff";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    };
    id: string;
    type: "session.handoff";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.truncation">;
    data: z.ZodObject<{
        tokenLimit: z.ZodNumber;
        preTruncationTokensInMessages: z.ZodNumber;
        preTruncationMessagesLength: z.ZodNumber;
        postTruncationTokensInMessages: z.ZodNumber;
        postTruncationMessagesLength: z.ZodNumber;
        tokensRemovedDuringTruncation: z.ZodNumber;
        messagesRemovedDuringTruncation: z.ZodNumber;
        performedBy: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    }, {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    };
    id: string;
    type: "session.truncation";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    };
    id: string;
    type: "session.truncation";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"session.snapshot_rewind">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        /** The event ID that was rewound to (events after this were removed) */
        upToEventId: z.ZodString;
        /** Number of events that were removed */
        eventsRemoved: z.ZodNumber;
    }, "strip", z.ZodTypeAny, {
        eventsRemoved: number;
        upToEventId: string;
    }, {
        eventsRemoved: number;
        upToEventId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        eventsRemoved: number;
        upToEventId: string;
    };
    id: string;
    ephemeral: true;
    type: "session.snapshot_rewind";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        eventsRemoved: number;
        upToEventId: string;
    };
    id: string;
    ephemeral: true;
    type: "session.snapshot_rewind";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"session.usage_info">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        tokenLimit: z.ZodNumber;
        currentTokens: z.ZodNumber;
        messagesLength: z.ZodNumber;
    }, "strip", z.ZodTypeAny, {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    }, {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    };
    id: string;
    ephemeral: true;
    type: "session.usage_info";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    };
    id: string;
    ephemeral: true;
    type: "session.usage_info";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.compaction_start">;
    data: z.ZodObject<{}, "strip", z.ZodTypeAny, {}, {}>;
}, "strip", z.ZodTypeAny, {
    data: {};
    id: string;
    type: "session.compaction_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {};
    id: string;
    type: "session.compaction_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.compaction_complete">;
    data: z.ZodObject<{
        success: z.ZodBoolean;
        error: z.ZodOptional<z.ZodString>;
        preCompactionTokens: z.ZodOptional<z.ZodNumber>;
        postCompactionTokens: z.ZodOptional<z.ZodNumber>;
        preCompactionMessagesLength: z.ZodOptional<z.ZodNumber>;
        messagesRemoved: z.ZodOptional<z.ZodNumber>;
        tokensRemoved: z.ZodOptional<z.ZodNumber>;
        summaryContent: z.ZodOptional<z.ZodString>;
        compactionTokensUsed: z.ZodOptional<z.ZodObject<{
            input: z.ZodNumber;
            output: z.ZodNumber;
            cachedInput: z.ZodNumber;
        }, "strip", z.ZodTypeAny, {
            input: number;
            output: number;
            cachedInput: number;
        }, {
            input: number;
            output: number;
            cachedInput: number;
        }>>;
    }, "strip", z.ZodTypeAny, {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    }, {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    };
    id: string;
    type: "session.compaction_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        success: boolean;
        error?: string | undefined;
        preCompactionTokens?: number | undefined;
        postCompactionTokens?: number | undefined;
        preCompactionMessagesLength?: number | undefined;
        messagesRemoved?: number | undefined;
        tokensRemoved?: number | undefined;
        summaryContent?: string | undefined;
        compactionTokensUsed?: {
            input: number;
            output: number;
            cachedInput: number;
        } | undefined;
    };
    id: string;
    type: "session.compaction_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"user.message">;
    data: z.ZodObject<{
        content: z.ZodString;
        transformedContent: z.ZodOptional<z.ZodString>;
        attachments: z.ZodOptional<z.ZodArray<z.ZodDiscriminatedUnion<"type", [z.ZodObject<{
            path: z.ZodString;
            displayName: z.ZodString;
        } & {
            type: z.ZodLiteral<"file">;
        }, "strip", z.ZodTypeAny, {
            type: "file";
            path: string;
            displayName: string;
        }, {
            type: "file";
            path: string;
            displayName: string;
        }>, z.ZodObject<{
            path: z.ZodString;
            displayName: z.ZodString;
        } & {
            type: z.ZodLiteral<"directory">;
        }, "strip", z.ZodTypeAny, {
            type: "directory";
            path: string;
            displayName: string;
        }, {
            type: "directory";
            path: string;
            displayName: string;
        }>, z.ZodObject<{
            type: z.ZodLiteral<"selection">;
            filePath: z.ZodString;
            displayName: z.ZodString;
            text: z.ZodString;
            selection: z.ZodObject<{
                start: z.ZodObject<{
                    line: z.ZodNumber;
                    character: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    line: number;
                    character: number;
                }, {
                    line: number;
                    character: number;
                }>;
                end: z.ZodObject<{
                    line: z.ZodNumber;
                    character: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    line: number;
                    character: number;
                }, {
                    line: number;
                    character: number;
                }>;
            }, "strip", z.ZodTypeAny, {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            }, {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            }>;
        }, "strip", z.ZodTypeAny, {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        }, {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        }>]>, "many">>;
        source: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    }, {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    };
    id: string;
    type: "user.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    };
    id: string;
    type: "user.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"pending_messages.modified">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{}, "strip", z.ZodTypeAny, {}, {}>;
}, "strip", z.ZodTypeAny, {
    data: {};
    id: string;
    ephemeral: true;
    type: "pending_messages.modified";
    timestamp: string;
    parentId: string | null;
}, {
    data: {};
    id: string;
    ephemeral: true;
    type: "pending_messages.modified";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.turn_start">;
    data: z.ZodObject<{
        turnId: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        turnId: string;
    }, {
        turnId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.intent">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        intent: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        intent: string;
    }, {
        intent: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        intent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.intent";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        intent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.intent";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.reasoning">;
    data: z.ZodObject<{
        reasoningId: z.ZodString;
        content: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        content: string;
        reasoningId: string;
    }, {
        content: string;
        reasoningId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        content: string;
        reasoningId: string;
    };
    id: string;
    type: "assistant.reasoning";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        content: string;
        reasoningId: string;
    };
    id: string;
    type: "assistant.reasoning";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.reasoning_delta">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        reasoningId: z.ZodString;
        deltaContent: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        reasoningId: string;
        deltaContent: string;
    }, {
        reasoningId: string;
        deltaContent: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        reasoningId: string;
        deltaContent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.reasoning_delta";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        reasoningId: string;
        deltaContent: string;
    };
    id: string;
    ephemeral: true;
    type: "assistant.reasoning_delta";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.message">;
    data: z.ZodObject<{
        messageId: z.ZodString;
        content: z.ZodString;
        toolRequests: z.ZodOptional<z.ZodArray<z.ZodObject<{
            toolCallId: z.ZodString;
            name: z.ZodString;
            arguments: z.ZodUnknown;
            type: z.ZodOptional<z.ZodEnum<["function", "custom"]>>;
        }, "strip", z.ZodTypeAny, {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }, {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }>, "many">>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    }, {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "assistant.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        content: string;
        messageId: string;
        toolRequests?: {
            name: string;
            toolCallId: string;
            type?: "function" | "custom" | undefined;
            arguments?: unknown;
        }[] | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "assistant.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.message_delta">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        messageId: z.ZodString;
        deltaContent: z.ZodString;
        totalResponseSizeBytes: z.ZodOptional<z.ZodNumber>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    }, {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.message_delta";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        deltaContent: string;
        messageId: string;
        parentToolCallId?: string | undefined;
        totalResponseSizeBytes?: number | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.message_delta";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"assistant.turn_end">;
    data: z.ZodObject<{
        turnId: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        turnId: string;
    }, {
        turnId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        turnId: string;
    };
    id: string;
    type: "assistant.turn_end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"assistant.usage">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        model: z.ZodOptional<z.ZodString>;
        inputTokens: z.ZodOptional<z.ZodNumber>;
        outputTokens: z.ZodOptional<z.ZodNumber>;
        cacheReadTokens: z.ZodOptional<z.ZodNumber>;
        cacheWriteTokens: z.ZodOptional<z.ZodNumber>;
        cost: z.ZodOptional<z.ZodNumber>;
        duration: z.ZodOptional<z.ZodNumber>;
        initiator: z.ZodOptional<z.ZodString>;
        apiCallId: z.ZodOptional<z.ZodString>;
        providerCallId: z.ZodOptional<z.ZodString>;
        quotaSnapshots: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodObject<{
            isUnlimitedEntitlement: z.ZodBoolean;
            entitlementRequests: z.ZodNumber;
            usedRequests: z.ZodNumber;
            usageAllowedWithExhaustedQuota: z.ZodBoolean;
            overage: z.ZodNumber;
            overageAllowedWithExhaustedQuota: z.ZodBoolean;
            remainingPercentage: z.ZodNumber;
            resetDate: z.ZodOptional<z.ZodDate>;
        }, "strip", z.ZodTypeAny, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }>>>;
    }, "strip", z.ZodTypeAny, {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    }, {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.usage";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        model?: string | undefined;
        duration?: number | undefined;
        inputTokens?: number | undefined;
        outputTokens?: number | undefined;
        cacheReadTokens?: number | undefined;
        cacheWriteTokens?: number | undefined;
        cost?: number | undefined;
        initiator?: string | undefined;
        apiCallId?: string | undefined;
        providerCallId?: string | undefined;
        quotaSnapshots?: Record<string, {
            isUnlimitedEntitlement: boolean;
            entitlementRequests: number;
            usedRequests: number;
            usageAllowedWithExhaustedQuota: boolean;
            overage: number;
            overageAllowedWithExhaustedQuota: boolean;
            remainingPercentage: number;
            resetDate?: Date | undefined;
        }> | undefined;
    };
    id: string;
    ephemeral: true;
    type: "assistant.usage";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"abort">;
    data: z.ZodObject<{
        reason: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        reason: string;
    }, {
        reason: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        reason: string;
    };
    id: string;
    type: "abort";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        reason: string;
    };
    id: string;
    type: "abort";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"tool.user_requested">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        toolName: z.ZodString;
        arguments: z.ZodUnknown;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    }, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    };
    id: string;
    type: "tool.user_requested";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    };
    id: string;
    type: "tool.user_requested";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"tool.execution_start">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        toolName: z.ZodString;
        arguments: z.ZodUnknown;
        mcpServerName: z.ZodOptional<z.ZodString>;
        mcpToolName: z.ZodOptional<z.ZodString>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    }, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "tool.execution_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "tool.execution_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"tool.execution_partial_result">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        partialOutput: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        partialOutput: string;
    }, {
        toolCallId: string;
        partialOutput: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        partialOutput: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_partial_result";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        toolCallId: string;
        partialOutput: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_partial_result";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"tool.execution_progress">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        progressMessage: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        progressMessage: string;
    }, {
        toolCallId: string;
        progressMessage: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        progressMessage: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_progress";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        toolCallId: string;
        progressMessage: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_progress";
    timestamp: string;
    parentId: string | null;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"tool.execution_complete">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        success: z.ZodBoolean;
        isUserRequested: z.ZodOptional<z.ZodBoolean>;
        result: z.ZodOptional<z.ZodObject<{
            /**
             * Tool result content sent to LLM for chat completion.
             * Typically concise/truncated for token efficiency.
             * Populated from: textResultForLlm || sessionLog
             */
            content: z.ZodString;
            /**
             * Detailed tool result for UI/timeline display.
             * Preserves full content like diffs. Optional - falls back to content.
             * Populated from: sessionLog || textResultForLlm
             */
            detailedContent: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            content: string;
            detailedContent?: string | undefined;
        }, {
            content: string;
            detailedContent?: string | undefined;
        }>>;
        error: z.ZodOptional<z.ZodObject<{
            message: z.ZodString;
            code: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            message: string;
            code?: string | undefined;
        }, {
            message: string;
            code?: string | undefined;
        }>>;
        toolTelemetry: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    }, {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    };
    id: string;
    type: "tool.execution_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    };
    id: string;
    type: "tool.execution_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.started">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        agentName: z.ZodString;
        agentDisplayName: z.ZodString;
        agentDescription: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    }, {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    };
    id: string;
    type: "subagent.started";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    };
    id: string;
    type: "subagent.started";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.completed">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        agentName: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        agentName: string;
    }, {
        toolCallId: string;
        agentName: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.completed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.completed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.failed">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        agentName: z.ZodString;
        error: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        error: string;
        toolCallId: string;
        agentName: string;
    }, {
        error: string;
        toolCallId: string;
        agentName: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        error: string;
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.failed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        error: string;
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.failed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.selected">;
    data: z.ZodObject<{
        agentName: z.ZodString;
        agentDisplayName: z.ZodString;
        tools: z.ZodNullable<z.ZodArray<z.ZodString, "many">>;
    }, "strip", z.ZodTypeAny, {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    }, {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    };
    id: string;
    type: "subagent.selected";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    };
    id: string;
    type: "subagent.selected";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"hook.start">;
    data: z.ZodObject<{
        hookInvocationId: z.ZodString;
        hookType: z.ZodString;
        input: z.ZodUnknown;
    }, "strip", z.ZodTypeAny, {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    }, {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    };
    id: string;
    type: "hook.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        hookInvocationId: string;
        hookType: string;
        input?: unknown;
    };
    id: string;
    type: "hook.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"hook.end">;
    data: z.ZodObject<{
        hookInvocationId: z.ZodString;
        hookType: z.ZodString;
        output: z.ZodUnknown;
        success: z.ZodBoolean;
        error: z.ZodOptional<z.ZodObject<{
            message: z.ZodString;
            stack: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            message: string;
            stack?: string | undefined;
        }, {
            message: string;
            stack?: string | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    }, {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    };
    id: string;
    type: "hook.end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        success: boolean;
        hookInvocationId: string;
        hookType: string;
        error?: {
            message: string;
            stack?: string | undefined;
        } | undefined;
        output?: unknown;
    };
    id: string;
    type: "hook.end";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>, z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"system.message">;
    data: z.ZodObject<{
        content: z.ZodString;
        role: z.ZodEnum<["system", "developer"]>;
        name: z.ZodOptional<z.ZodString>;
        metadata: z.ZodOptional<z.ZodObject<{
            promptVersion: z.ZodOptional<z.ZodString>;
            variables: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
        }, "strip", z.ZodTypeAny, {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        }, {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    }, {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    };
    id: string;
    type: "system.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    };
    id: string;
    type: "system.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>]>;

/**
 * All possible session event type strings
 */
export declare type SessionEventType = SessionEvent["type"];

export declare type SessionHandoffEvent = z.infer<typeof SessionHandoffEventSchema>;

/**
 * Remote session handed off to local
 */
declare const SessionHandoffEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.handoff">;
    data: z.ZodObject<{
        handoffTime: z.ZodString;
        sourceType: z.ZodEnum<["remote", "local"]>;
        repository: z.ZodOptional<z.ZodObject<{
            owner: z.ZodString;
            name: z.ZodString;
            branch: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            owner: string;
            name: string;
            branch?: string | undefined;
        }, {
            owner: string;
            name: string;
            branch?: string | undefined;
        }>>;
        context: z.ZodOptional<z.ZodString>;
        summary: z.ZodOptional<z.ZodString>;
        remoteSessionId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    }, {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    };
    id: string;
    type: "session.handoff";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        handoffTime: string;
        sourceType: "local" | "remote";
        summary?: string | undefined;
        context?: string | undefined;
        repository?: {
            owner: string;
            name: string;
            branch?: string | undefined;
        } | undefined;
        remoteSessionId?: string | undefined;
    };
    id: string;
    type: "session.handoff";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SessionIdleEvent = z.infer<typeof SessionIdleEventSchema>;

/**
 * Session initialization event (first line of JSONL file)
 */
declare const SessionIdleEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"session.idle">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{}, "strip", z.ZodTypeAny, {}, {}>;
}, "strip", z.ZodTypeAny, {
    data: {};
    id: string;
    ephemeral: true;
    type: "session.idle";
    timestamp: string;
    parentId: string | null;
}, {
    data: {};
    id: string;
    ephemeral: true;
    type: "session.idle";
    timestamp: string;
    parentId: string | null;
}>;

export declare type SessionImportLegacyEvent = z.infer<typeof SessionImportLegacyEventSchema>;

/**
 * Legacy session imported (wraps entire legacy JSON)
 */
declare const SessionImportLegacyEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.import_legacy">;
    data: z.ZodObject<{
        legacySession: z.ZodObject<{
            sessionId: z.ZodString;
            startTime: z.ZodDate;
            chatMessages: z.ZodArray<z.ZodUnion<[z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, "many">]>;
                role: z.ZodLiteral<"developer">;
                name: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }, {
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }>, z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, "many">]>;
                role: z.ZodLiteral<"system">;
                name: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }, {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            }>, z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodUnion<[z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"image_url">;
                    image_url: z.ZodObject<{
                        url: z.ZodString;
                        detail: z.ZodOptional<z.ZodEnum<["auto", "low", "high"]>>;
                    }, "strip", z.ZodTypeAny, {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    }, {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                }, {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"input_audio">;
                    input_audio: z.ZodObject<{
                        data: z.ZodString;
                        format: z.ZodUnion<[z.ZodLiteral<"wav">, z.ZodLiteral<"mp3">]>;
                    }, "strip", z.ZodTypeAny, {
                        data: string;
                        format: "wav" | "mp3";
                    }, {
                        data: string;
                        format: "wav" | "mp3";
                    }>;
                }, "strip", z.ZodTypeAny, {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                }, {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"file">;
                    file: z.ZodObject<{
                        file_date: z.ZodOptional<z.ZodString>;
                        file_id: z.ZodOptional<z.ZodString>;
                        filename: z.ZodOptional<z.ZodString>;
                    }, "strip", z.ZodTypeAny, {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    }, {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                }, {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                }>]>, "many">]>;
                role: z.ZodLiteral<"user">;
                name: z.ZodOptional<z.ZodString>;
            }, "strip", z.ZodTypeAny, {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            }, {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            }>, z.ZodObject<{
                content: z.ZodOptional<z.ZodNullable<z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodUnion<[z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"refusal">;
                    refusal: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    refusal: string;
                    type: "refusal";
                }, {
                    refusal: string;
                    type: "refusal";
                }>]>, "many">]>>>;
                role: z.ZodLiteral<"assistant">;
                name: z.ZodOptional<z.ZodString>;
                refusal: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                audio: z.ZodOptional<z.ZodNullable<z.ZodObject<{
                    id: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    id: string;
                }, {
                    id: string;
                }>>>;
                function_call: z.ZodOptional<z.ZodNullable<z.ZodObject<{
                    name: z.ZodString;
                    arguments: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    name: string;
                    arguments: string;
                }, {
                    name: string;
                    arguments: string;
                }>>>;
                tool_calls: z.ZodOptional<z.ZodArray<z.ZodUnion<[z.ZodObject<{
                    id: z.ZodString;
                    type: z.ZodLiteral<"function">;
                    function: z.ZodObject<{
                        name: z.ZodString;
                        arguments: z.ZodString;
                    }, "strip", z.ZodTypeAny, {
                        name: string;
                        arguments: string;
                    }, {
                        name: string;
                        arguments: string;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                }, {
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                }>, z.ZodObject<{
                    id: z.ZodString;
                    type: z.ZodLiteral<"custom">;
                    custom: z.ZodObject<{
                        name: z.ZodString;
                        input: z.ZodString;
                    }, "strip", z.ZodTypeAny, {
                        input: string;
                        name: string;
                    }, {
                        input: string;
                        name: string;
                    }>;
                }, "strip", z.ZodTypeAny, {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                }, {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                }>]>, "many">>;
            }, "strip", z.ZodTypeAny, {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            }, {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            }>, z.ZodObject<{
                content: z.ZodUnion<[z.ZodString, z.ZodArray<z.ZodObject<{
                    type: z.ZodLiteral<"text">;
                    text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    type: "text";
                    text: string;
                }, {
                    type: "text";
                    text: string;
                }>, "many">]>;
                role: z.ZodLiteral<"tool">;
                tool_call_id: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            }, {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            }>, z.ZodObject<{
                content: z.ZodNullable<z.ZodString>;
                role: z.ZodLiteral<"function">;
                name: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                name: string;
                role: "function";
                content: string | null;
            }, {
                name: string;
                role: "function";
                content: string | null;
            }>]>, "many">;
            timeline: z.ZodArray<z.ZodIntersection<z.ZodUnion<[z.ZodObject<{
                type: z.ZodLiteral<"copilot">;
                text: z.ZodString;
                isStreaming: z.ZodOptional<z.ZodBoolean>;
            }, "strip", z.ZodTypeAny, {
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            }, {
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"error">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "error";
                text: string;
            }, {
                type: "error";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"info">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "info";
                text: string;
            }, {
                type: "info";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"warning">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "warning";
                text: string;
            }, {
                type: "warning";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"user">;
                text: z.ZodString;
            }, "strip", z.ZodTypeAny, {
                type: "user";
                text: string;
            }, {
                type: "user";
                text: string;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"tool_call_requested">;
                callId: z.ZodString;
                name: z.ZodString;
                toolTitle: z.ZodOptional<z.ZodString>;
                intentionSummary: z.ZodNullable<z.ZodString>;
                arguments: z.ZodUnion<[z.ZodUnion<[z.ZodObject<{
                    command: z.ZodString;
                    description: z.ZodString;
                    timeout: z.ZodOptional<z.ZodNumber>;
                    sessionId: z.ZodOptional<z.ZodString>;
                    async: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    input: z.ZodString;
                    delay: z.ZodOptional<z.ZodNumber>;
                }, "strip", z.ZodTypeAny, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    delay: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                    delay: number;
                }, {
                    sessionId: string;
                    delay: number;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                }, {
                    sessionId: string;
                }>]>, z.ZodDiscriminatedUnion<"command", [z.ZodObject<{
                    command: z.ZodLiteral<"view">;
                    path: z.ZodString;
                    view_range: z.ZodOptional<z.ZodTuple<[z.ZodNumber, z.ZodNumber], null>>;
                }, "strip", z.ZodTypeAny, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"create">;
                    path: z.ZodString;
                    file_text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "create";
                    path: string;
                    file_text: string;
                }, {
                    command: "create";
                    path: string;
                    file_text: string;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"str_replace">;
                    path: z.ZodString;
                    new_str: z.ZodOptional<z.ZodString>;
                    old_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"insert">;
                    path: z.ZodString;
                    insert_line: z.ZodNumber;
                    new_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }>]>, z.ZodUnknown]>;
                partialOutput: z.ZodOptional<z.ZodString>;
                isHidden: z.ZodOptional<z.ZodBoolean>;
                isAlwaysExpanded: z.ZodOptional<z.ZodBoolean>;
                showNoContent: z.ZodOptional<z.ZodBoolean>;
            }, "strip", z.ZodTypeAny, {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }, {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }>, z.ZodObject<{
                type: z.ZodLiteral<"tool_call_completed">;
                callId: z.ZodString;
                name: z.ZodString;
                toolTitle: z.ZodOptional<z.ZodString>;
                intentionSummary: z.ZodNullable<z.ZodString>;
                result: z.ZodUnion<[z.ZodObject<{
                    type: z.ZodLiteral<"success">;
                    log: z.ZodString;
                    detailedLog: z.ZodOptional<z.ZodString>;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                }, {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"failure">;
                    log: z.ZodString;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                }, {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"rejected">;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    type: "rejected";
                    markdown?: boolean | undefined;
                }, {
                    type: "rejected";
                    markdown?: boolean | undefined;
                }>, z.ZodObject<{
                    type: z.ZodLiteral<"denied">;
                    log: z.ZodString;
                    markdown: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                }, {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                }>]>;
                arguments: z.ZodUnion<[z.ZodUnion<[z.ZodObject<{
                    command: z.ZodString;
                    description: z.ZodString;
                    timeout: z.ZodOptional<z.ZodNumber>;
                    sessionId: z.ZodOptional<z.ZodString>;
                    async: z.ZodOptional<z.ZodBoolean>;
                }, "strip", z.ZodTypeAny, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }, {
                    command: string;
                    description: string;
                    sessionId?: string | undefined;
                    timeout?: number | undefined;
                    async?: boolean | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    input: z.ZodString;
                    delay: z.ZodOptional<z.ZodNumber>;
                }, "strip", z.ZodTypeAny, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }, {
                    input: string;
                    sessionId: string;
                    delay?: number | undefined;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                    delay: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                    delay: number;
                }, {
                    sessionId: string;
                    delay: number;
                }>, z.ZodObject<{
                    sessionId: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    sessionId: string;
                }, {
                    sessionId: string;
                }>]>, z.ZodDiscriminatedUnion<"command", [z.ZodObject<{
                    command: z.ZodLiteral<"view">;
                    path: z.ZodString;
                    view_range: z.ZodOptional<z.ZodTuple<[z.ZodNumber, z.ZodNumber], null>>;
                }, "strip", z.ZodTypeAny, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }, {
                    command: "view";
                    path: string;
                    view_range?: [number, number] | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"create">;
                    path: z.ZodString;
                    file_text: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "create";
                    path: string;
                    file_text: string;
                }, {
                    command: "create";
                    path: string;
                    file_text: string;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"str_replace">;
                    path: z.ZodString;
                    new_str: z.ZodOptional<z.ZodString>;
                    old_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }, {
                    command: "str_replace";
                    path: string;
                    old_str: string;
                    new_str?: string | undefined;
                }>, z.ZodObject<{
                    command: z.ZodLiteral<"insert">;
                    path: z.ZodString;
                    insert_line: z.ZodNumber;
                    new_str: z.ZodString;
                }, "strip", z.ZodTypeAny, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }, {
                    command: "insert";
                    path: string;
                    new_str: string;
                    insert_line: number;
                }>]>, z.ZodUnknown]>;
                isHidden: z.ZodOptional<z.ZodBoolean>;
                isAlwaysExpanded: z.ZodOptional<z.ZodBoolean>;
                showNoContent: z.ZodOptional<z.ZodBoolean>;
            }, "strip", z.ZodTypeAny, {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }, {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }>]>, z.ZodObject<{
                id: z.ZodString;
                timestamp: z.ZodDate;
            }, "strip", z.ZodTypeAny, {
                id: string;
                timestamp: Date;
            }, {
                id: string;
                timestamp: Date;
            }>>, "many">;
            selectedModel: z.ZodOptional<z.ZodEnum<["claude-sonnet-4.5", "claude-haiku-4.5", "claude-opus-4.5", "claude-sonnet-4", "gpt-5.2-codex", "gpt-5.1-codex-max", "gpt-5.1-codex", "gpt-5.2", "gpt-5.1", "gpt-5", "gpt-5.1-codex-mini", "gpt-5-mini", "gpt-4.1", "gemini-3-pro-preview"]>>;
        }, "strip", z.ZodTypeAny, {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        }, {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        }>;
        importTime: z.ZodString;
        sourceFile: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    }, {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    };
    id: string;
    type: "session.import_legacy";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        legacySession: {
            sessionId: string;
            startTime: Date;
            chatMessages: ({
                role: "developer";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "system";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                name?: string | undefined;
            } | {
                role: "user";
                content: string | ({
                    type: "text";
                    text: string;
                } | {
                    type: "image_url";
                    image_url: {
                        url: string;
                        detail?: "auto" | "low" | "high" | undefined;
                    };
                } | {
                    type: "input_audio";
                    input_audio: {
                        data: string;
                        format: "wav" | "mp3";
                    };
                } | {
                    type: "file";
                    file: {
                        file_date?: string | undefined;
                        file_id?: string | undefined;
                        filename?: string | undefined;
                    };
                })[];
                name?: string | undefined;
            } | {
                role: "assistant";
                name?: string | undefined;
                tool_calls?: ({
                    function: {
                        name: string;
                        arguments: string;
                    };
                    id: string;
                    type: "function";
                } | {
                    id: string;
                    custom: {
                        input: string;
                        name: string;
                    };
                    type: "custom";
                })[] | undefined;
                audio?: {
                    id: string;
                } | null | undefined;
                content?: string | ({
                    type: "text";
                    text: string;
                } | {
                    refusal: string;
                    type: "refusal";
                })[] | null | undefined;
                function_call?: {
                    name: string;
                    arguments: string;
                } | null | undefined;
                refusal?: string | null | undefined;
            } | {
                role: "tool";
                content: string | {
                    type: "text";
                    text: string;
                }[];
                tool_call_id: string;
            } | {
                name: string;
                role: "function";
                content: string | null;
            })[];
            timeline: (({
                type: "copilot";
                text: string;
                isStreaming?: boolean | undefined;
            } | {
                type: "error";
                text: string;
            } | {
                type: "info";
                text: string;
            } | {
                type: "warning";
                text: string;
            } | {
                type: "user";
                text: string;
            } | {
                name: string;
                type: "tool_call_requested";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                partialOutput?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            } | {
                result: {
                    log: string;
                    type: "success";
                    detailedLog?: string | undefined;
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "failure";
                    markdown?: boolean | undefined;
                } | {
                    type: "rejected";
                    markdown?: boolean | undefined;
                } | {
                    log: string;
                    type: "denied";
                    markdown?: boolean | undefined;
                };
                name: string;
                type: "tool_call_completed";
                callId: string;
                intentionSummary: string | null;
                arguments?: unknown;
                toolTitle?: string | undefined;
                isHidden?: boolean | undefined;
                isAlwaysExpanded?: boolean | undefined;
                showNoContent?: boolean | undefined;
            }) & {
                id: string;
                timestamp: Date;
            })[];
            selectedModel?: "gpt-5" | "gpt-5-mini" | "gpt-4.1" | "claude-sonnet-4.5" | "claude-opus-4.5" | "claude-haiku-4.5" | "claude-sonnet-4" | "gpt-5.2-codex" | "gpt-5.1-codex-max" | "gpt-5.1-codex" | "gpt-5.2" | "gpt-5.1" | "gpt-5.1-codex-mini" | "gemini-3-pro-preview" | undefined;
        };
        importTime: string;
        sourceFile: string;
    };
    id: string;
    type: "session.import_legacy";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SessionInfoEvent = z.infer<typeof SessionInfoEventSchema>;

/**
 * Informational message (for timeline/UI display)
 */
declare const SessionInfoEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.info">;
    data: z.ZodObject<{
        infoType: z.ZodString;
        message: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        message: string;
        infoType: string;
    }, {
        message: string;
        infoType: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        message: string;
        infoType: string;
    };
    id: string;
    type: "session.info";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        message: string;
        infoType: string;
    };
    id: string;
    type: "session.info";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

declare type SessionLogsContent = SessionsChatCompletionMessageParam | SessionsChatCompletionChunk;

declare type SessionLogsContents = SessionLogsContent[];

/**
 * SessionManager interface
 */
export declare interface SessionManager<TSessionMetadata extends SessionMetadata = SessionMetadata, TSession extends Session<TSessionMetadata> = Session<TSessionMetadata>> {
    createSession(sessionOptions: SessionOptions): Promise<TSession>;
    getSession(options: {
        sessionId: string;
    }): Promise<TSession | undefined>;
    getLastSession(): Promise<TSession | undefined>;
    getLastSessionId(): Promise<string | undefined>;
    listSessions(): Promise<TSessionMetadata[]>;
    saveSession(session: TSession): Promise<void>;
    deleteSession(sessionId: string): Promise<void>;
    closeSession(sessionId: string): Promise<void>;
}

/**
 * SessionManager options - same as SessionOptions but used for creating the manager
 */
export declare type SessionManagerOptions = {
    logger?: RunnerLogger;
    integrationId?: string;
};

export declare interface SessionMetadata {
    readonly sessionId: string;
    readonly startTime: Date;
    readonly modifiedTime: Date;
    readonly summary?: string;
    readonly isRemote: boolean;
    /** Most recent working directory context (from last start or resume) */
    readonly context?: SessionContext;
}

export declare type SessionModelChangeEvent = z.infer<typeof SessionModelChangeEventSchema>;

/**
 * Model selection changed mid-session
 */
declare const SessionModelChangeEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.model_change">;
    data: z.ZodObject<{
        previousModel: z.ZodOptional<z.ZodString>;
        newModel: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        newModel: string;
        previousModel?: string | undefined;
    }, {
        newModel: string;
        previousModel?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        newModel: string;
        previousModel?: string | undefined;
    };
    id: string;
    type: "session.model_change";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        newModel: string;
        previousModel?: string | undefined;
    };
    id: string;
    type: "session.model_change";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare interface SessionOptions extends Partial<SessionMetadata> {
    clientName?: string;
    model?: SupportedModel;
    integrationId?: string;
    /**
     * Custom API provider configuration (BYOK - Bring Your Own Key).
     * When set, bypasses Copilot API authentication and uses this provider instead.
     */
    provider?: ProviderConfig;
    featureFlags?: FeatureFlags;
    availableTools?: string[];
    excludedTools?: string[];
    executeToolsInParallel?: boolean;
    shellConfig?: ShellConfig;
    requestPermission?: (permissionRequest: PermissionRequest) => Promise<PermissionRequestResult>;
    skillDirectories?: string[];
    disabledSkills?: Set<string>;
    installedPlugins?: InstalledPlugin[];
    mcpServers?: Record<string, MCPServerConfig>;
    mcpHost?: McpHost;
    customAgents?: SweCustomAgent[];
    selectedCustomAgent?: SweCustomAgent;
    organizationCustomInstructions?: string;
    skipCustomInstructions?: boolean;
    systemMessage?: SystemMessageConfig;
    hooks?: QueryHooks;
    externalToolDefinitions?: ExternalToolDefinition[];
    externalToolDispatcher?: ExternalToolDispatcher;
    trajectoryFile?: string;
    eventsLogDirectory?: string;
    workingDirectory?: string;
    /**
     * Repository name for the session context.
     * Used for memory service scoping, code search, and other repository-aware features.
     * Format: "owner/repo" (e.g., "github/copilot-cli")
     */
    repositoryName?: string;
    authInfo?: AuthInfo;
    copilotUrl?: string;
    enableStreaming?: boolean;
    largeOutput?: LargeToolOutputConfig;
    /** Callback invoked when a file is created. Useful for refreshing file caches. */
    onFileCreated?: (path: string) => void;
    /** Callback invoked when a warning should be emitted to the timeline. */
    onWarning?: (message: string) => void;
    /** Callback to request user input from the UI. Enables ask_user tool in interactive mode. */
    requestUserInput?: (request: {
        question: string;
        choices?: string[];
        allowFreeform?: boolean;
    }) => Promise<{
        answer: string;
        wasFreeform: boolean;
    }>;
    /** Whether ask_user is explicitly disabled (autonomous mode). When true, system prompt encourages independent action. */
    askUserDisabled?: boolean;
    /** Whether the CLI is running in interactive mode. Defaults to true. When false, uses non-interactive identity and excludes plan mode instructions. */
    runningInInteractiveMode?: boolean;
    /**
     * Custom configuration directory for the session.
     * When set, overrides the default Copilot config directory (~/.copilot or $XDG_CONFIG_HOME/.copilot).
     */
    configDir?: string;
    /**
     * Runtime settings for persistence and workspace resolution.
     * Used to resolve config/state paths consistently.
     */
    runtimeSettings?: RuntimeSettings;
    /**
     * Infinite session configuration for persistent workspaces and automatic compaction.
     * When enabled, sessions automatically manage context limits and persist state.
     * Can be set to `true` for defaults, `false` to disable, or a config object for fine-tuning.
     * @default { enabled: true }
     */
    infiniteSessions?: InfiniteSessionConfig | boolean;
}

export declare type SessionResumeEvent = z.infer<typeof SessionResumeEventSchema>;

/**
 * Session resumed from disk
 */
declare const SessionResumeEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.resume">;
    data: z.ZodObject<{
        resumeTime: z.ZodString;
        eventCount: z.ZodNumber;
        context: z.ZodOptional<z.ZodObject<{
            cwd: z.ZodString;
            gitRoot: z.ZodOptional<z.ZodString>;
            repository: z.ZodOptional<z.ZodString>;
            branch: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    }, {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    };
    id: string;
    type: "session.resume";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        resumeTime: string;
        eventCount: number;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
    };
    id: string;
    type: "session.resume";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

declare type SessionsChatCompletionChunk = CopilotChatCompletionChunk & AgentIdBag;

declare type SessionsChatCompletionMessageParam = (SessionsUserMessageParam | ChatCompletionToolMessageParam) & AgentIdBag;

export declare type SessionSnapshotRewindEvent = z.infer<typeof SessionSnapshotRewindEventSchema>;

/**
 * Session rewind event - emitted when session events are rewound (e.g., during snapshot rollback)
 * This is ephemeral and used to signal the UI to reconstruct its state from the remaining events.
 */
declare const SessionSnapshotRewindEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"session.snapshot_rewind">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        /** The event ID that was rewound to (events after this were removed) */
        upToEventId: z.ZodString;
        /** Number of events that were removed */
        eventsRemoved: z.ZodNumber;
    }, "strip", z.ZodTypeAny, {
        eventsRemoved: number;
        upToEventId: string;
    }, {
        eventsRemoved: number;
        upToEventId: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        eventsRemoved: number;
        upToEventId: string;
    };
    id: string;
    ephemeral: true;
    type: "session.snapshot_rewind";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        eventsRemoved: number;
        upToEventId: string;
    };
    id: string;
    ephemeral: true;
    type: "session.snapshot_rewind";
    timestamp: string;
    parentId: string | null;
}>;

export declare type SessionStartEvent = z.infer<typeof SessionStartEventSchema>;

/**
 * Session initialization event (first line of JSONL file)
 */
declare const SessionStartEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.start">;
    data: z.ZodObject<{
        sessionId: z.ZodString;
        version: z.ZodNumber;
        producer: z.ZodString;
        copilotVersion: z.ZodString;
        startTime: z.ZodString;
        selectedModel: z.ZodOptional<z.ZodString>;
        context: z.ZodOptional<z.ZodObject<{
            cwd: z.ZodString;
            gitRoot: z.ZodOptional<z.ZodString>;
            repository: z.ZodOptional<z.ZodString>;
            branch: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }, {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    }, {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    };
    id: string;
    type: "session.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        version: number;
        sessionId: string;
        producer: string;
        copilotVersion: string;
        startTime: string;
        context?: {
            cwd: string;
            branch?: string | undefined;
            gitRoot?: string | undefined;
            repository?: string | undefined;
        } | undefined;
        selectedModel?: string | undefined;
    };
    id: string;
    type: "session.start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SessionStartHook = (input: SessionStartHookInput) => Promise<SessionStartHookOutput | void>;

/**
 * Session start hook types
 */
export declare interface SessionStartHookInput extends BaseHookInput {
    source: "startup" | "resume" | "new";
    initialPrompt?: string;
}

export declare interface SessionStartHookOutput {
    additionalContext?: string;
    modifiedConfig?: Record<string, unknown>;
}

declare type SessionsUserMessageParam = ChatCompletionUserMessageParam & {
    /**
     * The component which was the source of the user message.
     * - `jit-instruction`: The message was injected by something which adds automated instructions for the agent
     * - `command-{id}`: The message was injected as a result of a command with the given id
     * - `string`: Some other source
     */
    source?: string;
};

export declare type SessionTruncationEvent = z.infer<typeof SessionTruncationEventSchema>;

declare const SessionTruncationEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"session.truncation">;
    data: z.ZodObject<{
        tokenLimit: z.ZodNumber;
        preTruncationTokensInMessages: z.ZodNumber;
        preTruncationMessagesLength: z.ZodNumber;
        postTruncationTokensInMessages: z.ZodNumber;
        postTruncationMessagesLength: z.ZodNumber;
        tokensRemovedDuringTruncation: z.ZodNumber;
        messagesRemovedDuringTruncation: z.ZodNumber;
        performedBy: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    }, {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    };
    id: string;
    type: "session.truncation";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
        performedBy: string;
    };
    id: string;
    type: "session.truncation";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SessionUsageInfoEvent = z.infer<typeof SessionUsageInfoEventSchema>;

/**
 * Usage info event - emitted every turn to report current context window token usage.
 * Unlike truncation events, this is always emitted regardless of whether truncation occurred.
 */
declare const SessionUsageInfoEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"session.usage_info">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        tokenLimit: z.ZodNumber;
        currentTokens: z.ZodNumber;
        messagesLength: z.ZodNumber;
    }, "strip", z.ZodTypeAny, {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    }, {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    };
    id: string;
    ephemeral: true;
    type: "session.usage_info";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        tokenLimit: number;
        currentTokens: number;
        messagesLength: number;
    };
    id: string;
    ephemeral: true;
    type: "session.usage_info";
    timestamp: string;
    parentId: string | null;
}>;

declare class ShellConfig {
    /**
     * Used to vary behavior programmatically for different shell implementations.
     */
    readonly shellType: ShellType;
    /**
     * Used when describing the shell type to users or in prompts.
     */
    readonly displayName: string;
    /**
     * Name of a tool that invokes a shell command.
     */
    readonly shellToolName: string;
    /**
     * Name of a tool that reads output from a shell session.
     */
    readonly readShellToolName: string;
    /**
     * Name of a tool that sends input to a shell session.
     */
    readonly writeShellToolName: string;
    /**
     * Name of a tool that terminates a shell session.
     */
    readonly stopShellToolName: string;
    /**
     * Name of a tool that lists all active shell sessions.
     */
    readonly listShellsToolName: string;
    /**
     * Additional information to add to the tool description.
     */
    readonly descriptionLines: string[];
    /**
     * A function that assesses the safety of a script to be run in the shell.
     * The reason this exists as a pluggable abstraction is so that only the
     * CLI needs to take a compile-time dependency on tree-sitter, and CCA doesn't.
     */
    readonly assessScriptSafety: (script: string, onWarning?: (message: string) => void) => Promise<SafetyAssessment>;
    constructor(
    /**
     * Used to vary behavior programmatically for different shell implementations.
     */
    shellType: ShellType, 
    /**
     * Used when describing the shell type to users or in prompts.
     */
    displayName: string, 
    /**
     * Name of a tool that invokes a shell command.
     */
    shellToolName: string, 
    /**
     * Name of a tool that reads output from a shell session.
     */
    readShellToolName: string, 
    /**
     * Name of a tool that sends input to a shell session.
     */
    writeShellToolName: string, 
    /**
     * Name of a tool that terminates a shell session.
     */
    stopShellToolName: string, 
    /**
     * Name of a tool that lists all active shell sessions.
     */
    listShellsToolName: string, 
    /**
     * Additional information to add to the tool description.
     */
    descriptionLines: string[], 
    /**
     * A function that assesses the safety of a script to be run in the shell.
     * The reason this exists as a pluggable abstraction is so that only the
     * CLI needs to take a compile-time dependency on tree-sitter, and CCA doesn't.
     */
    assessScriptSafety?: (script: string, onWarning?: (message: string) => void) => Promise<SafetyAssessment>);
    withScriptSafetyAssessor(settings: RuntimeSettings | undefined, assessor: (settings: RuntimeSettings | undefined, shellType: ShellType, script: string, onWarning?: (message: string) => void) => Promise<SafetyAssessment>): ShellConfig;
    static readonly bash: ShellConfig;
    static readonly powerShell: ShellConfig;
}

/**
 * A permission request for executing shell commands.
 */
declare type ShellPermissionRequest = {
    readonly kind: "shell";
    /** The full command that the user is being asked to approve, e.g. `echo foo && find -exec ... && git push` */
    readonly fullCommandText: string;
    /** A concise summary of the user's intention, e.g. "Echo foo and find a file and then run git push" */
    readonly intention: string;
    /**
     * The commands that are being invoked in the shell invocation.
     *
     * As a special case, which might be better represented in the type system, if there were no parsed commands
     * e.g. `export VAR=value`, then this will have a single entry with identifier equal to the fullCommandText.
     */
    readonly commands: ReadonlyArray<Command>;
    /**
     * Possible file paths that the command might access.
     *
     * This is entirely heuristic, so it's pretty untrustworthy.
     */
    readonly possiblePaths: ReadonlyArray<PossiblePath>;
    /**
     * Possible URLs that the command might access.
     *
     * This is entirely heuristic, so it's pretty untrustworthy.
     */
    readonly possibleUrls: ReadonlyArray<PossibleUrl>;
    /**
     * Indicates whether any command in the script has redirection to write to a file.
     */
    readonly hasWriteFileRedirection: boolean;
    /**
     * If there are complicated constructs, then persistent approval is not supported.
     * e.g. `cat $(echo "foo")` should not be persistently approvable because it's hard
     * for the user to understand the implications.
     */
    readonly canOfferSessionApproval: boolean;
    /**
     * Optional warning message to display (e.g., when the shell parser is unavailable).
     */
    readonly warning?: string;
};

declare type ShellType = "bash" | "powershell";

declare interface SSETransportConfig {
    type: "sse";
    url: string;
    headers?: Record<string, string>;
    authProvider?: OAuthClientProvider;
}

declare interface StdioTransportConfig {
    type: "stdio";
    command: string;
    args: string[];
    env: Record<string, string>;
    stderr?: Writable;
    cwd?: string;
}

/**
 * Simplified streaming chunk context containing only the essential delta information
 * needed by processors. This avoids the complexity of converting between different
 * API formats (e.g., Responses API to ChatCompletion chunks).
 *
 * @deprecated In the future, we will move to a model where the individual API clients emit Core Runtime events, instead of the current model where ChatCompletions are the defacto interface.
 * Please avoid adding new streaming chunk processors, and instead consider if now is the right time to fix the abstraction gap between the ChatCompletionsClient and the DerivedResponsesClient.
 * Talk to @jmoseley or @mrayermannmsft for more context.
 */
declare type StreamingChunkContext = {
    /**
     * The streaming ID of the message.
     */
    streamingId: string;
    /**
     * Text content delta from this chunk.
     */
    content?: string;
    /**
     * Reasoning content delta from this chunk (chain-of-thought summaries).
     */
    reasoningContent?: string;
    /**
     * Arguments delta for the report_intent tool call. Processors can accumulate
     * these deltas to extract the intent once the JSON is complete.
     */
    reportIntentArguments?: string;
    /**
     * Approximate byte size of this chunk, calculated from content and all tool call data.
     */
    size: number;
};

export declare type SubagentCompletedEvent = z.infer<typeof SubagentCompletedEventSchema>;

/**
 * Subagent execution completes successfully
 */
declare const SubagentCompletedEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.completed">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        agentName: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        agentName: string;
    }, {
        toolCallId: string;
        agentName: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.completed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.completed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SubagentFailedEvent = z.infer<typeof SubagentFailedEventSchema>;

/**
 * Subagent execution fails
 */
declare const SubagentFailedEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.failed">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        agentName: z.ZodString;
        error: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        error: string;
        toolCallId: string;
        agentName: string;
    }, {
        error: string;
        toolCallId: string;
        agentName: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        error: string;
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.failed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        error: string;
        toolCallId: string;
        agentName: string;
    };
    id: string;
    type: "subagent.failed";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SubagentSelectedEvent = z.infer<typeof SubagentSelectedEventSchema>;

/**
 * Subagent selected for the session
 */
declare const SubagentSelectedEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.selected">;
    data: z.ZodObject<{
        agentName: z.ZodString;
        agentDisplayName: z.ZodString;
        tools: z.ZodNullable<z.ZodArray<z.ZodString, "many">>;
    }, "strip", z.ZodTypeAny, {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    }, {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    };
    id: string;
    type: "subagent.selected";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        tools: string[] | null;
        agentName: string;
        agentDisplayName: string;
    };
    id: string;
    type: "subagent.selected";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type SubagentStartedEvent = z.infer<typeof SubagentStartedEventSchema>;

/**
 * Subagent execution begins
 */
declare const SubagentStartedEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"subagent.started">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        agentName: z.ZodString;
        agentDisplayName: z.ZodString;
        agentDescription: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    }, {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    };
    id: string;
    type: "subagent.started";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        agentName: string;
        agentDisplayName: string;
        agentDescription: string;
    };
    id: string;
    type: "subagent.started";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

/** List of supported model IDs in order of precedence to be used as the default */
declare const SUPPORTED_MODELS: readonly ["claude-sonnet-4.5", "claude-haiku-4.5", "claude-opus-4.5", "claude-sonnet-4", "gpt-5.2-codex", "gpt-5.1-codex-max", "gpt-5.1-codex", "gpt-5.2", "gpt-5.1", "gpt-5", "gpt-5.1-codex-mini", "gpt-5-mini", "gpt-4.1", "gemini-3-pro-preview"];

declare type SupportedModel = (typeof SUPPORTED_MODELS)[number];

/**
 * Subset of src/types/clients/types.ts that is required to actually run
 * a custom agent.
 */
export declare type SweCustomAgent = {
    name: string;
    displayName: string;
    description: string;
    tools: string[] | null;
    prompt: () => Promise<string>;
    mcpServers?: Record<string, MCPServerConfig>;
    infer: boolean;
};

/**
 * Append mode: Use CLI foundation with optional appended content (default).
 */
declare interface SystemMessageAppendConfig {
    mode?: "append";
    /**
     * Additional instructions appended after SDK-managed sections.
     */
    content?: string;
}

/**
 * System message configuration for session creation.
 * - Append mode (default): SDK foundation + optional custom content
 * - Replace mode: Full control, caller provides entire system message
 */
declare type SystemMessageConfig = SystemMessageAppendConfig | SystemMessageReplaceConfig;

export declare type SystemMessageEvent = z.infer<typeof SystemMessageEventSchema>;

/**
 * System message/prompt
 */
declare const SystemMessageEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"system.message">;
    data: z.ZodObject<{
        content: z.ZodString;
        role: z.ZodEnum<["system", "developer"]>;
        name: z.ZodOptional<z.ZodString>;
        metadata: z.ZodOptional<z.ZodObject<{
            promptVersion: z.ZodOptional<z.ZodString>;
            variables: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
        }, "strip", z.ZodTypeAny, {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        }, {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        }>>;
    }, "strip", z.ZodTypeAny, {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    }, {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    };
    id: string;
    type: "system.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        role: "developer" | "system";
        content: string;
        name?: string | undefined;
        metadata?: {
            promptVersion?: string | undefined;
            variables?: Record<string, unknown> | undefined;
        } | undefined;
    };
    id: string;
    type: "system.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

/**
 * Replace mode: Use caller-provided system message entirely.
 * Removes all SDK guardrails including security restrictions.
 */
declare interface SystemMessageReplaceConfig {
    mode: "replace";
    /**
     * Complete system message content.
     * Replaces the entire SDK-managed system message.
     */
    content: string;
}

/**
 * Telemetry emitted by the runtime contains properties and metrics. These are non-sensitive pieces
 * of information. There are also restricted properties that must be used to store sensitive information.
 */
declare type Telemetry = {
    /**
     * Telemetry properties can be used to store string props.
     * WARNING: Do not put sensitive data here. Use restrictedProperties for that.
     */
    properties: Record<string, string | undefined>;
    /**
     * Restricted telemetry properties must be used to store sensitive string props. These props will only be available on the restricted kusto topics.
     * Nonnullable so it is harder to overlook.
     */
    restrictedProperties: Record<string, string | undefined>;
    /**
     * The name of the telemetry event associated with the emitted runtime event.
     */
    metrics: Record<string, number | undefined>;
};

/**
 * Alternatively telemetry can be emitted by an event which just contains telemetry. This is that type.
 *
 * You can use this type with our without generics. The generics help you to enforce what properties/metrics are on your event
 * more precisely and safely.
 */
declare type TelemetryEvent<EventT = string, TelemetryT extends Telemetry = Telemetry> = {
    kind: "telemetry";
    telemetry: EventTelemetry<EventT, TelemetryT>;
};

/** Represents a Token authentication information using in the SDK. */
declare type TokenAuthInfo = {
    readonly type: "token";
    readonly host: string;
    readonly token: string;
};

declare type Tool = {
    name: string;
    namespacedName: string;
    /** Original/display name for the MCP server (may contain "/" and/or "__"). */
    mcpServerName?: string;
    /** Raw MCP tool name as reported by the server. */
    mcpToolName?: string;
    /** Intended for UI and end-user contexts — optimized to be human-readable
     * and easily understood, even by those unfamiliar with domain-specific terminology.
     *
     * If not provided, the name should be used for display (except for Tool, where
     * annotations.title should be given precedence over using name, if present).
     */
    title: string;
    description: string;
    input_schema: ToolInputSchema;
    readOnly?: boolean;
    safeForTelemetry: {
        name: boolean;
        inputsNames: boolean;
    };
    filterMode?: ContentFilterMode;
};

declare type Tool_2<CallbackT extends ToolCallback = ToolCallback> = ToolMetadata & {
    /**
     * The type of the tool. Defaults to "function" if not specified.
     * - "function": Standard function tool with JSON Schema input
     * - "custom": Custom tool with grammar-based input format
     */
    type?: "function" | "custom";
    /**
     * The input format for custom tools. Only used when type is "custom".
     */
    format?: CustomToolInputFormat;
    /**
     * A human readable string summary of what this command intends to do if executed.
     *
     * If not set, no summarised intention should be assumed by the caller.
     */
    summariseIntention?: (input: unknown) => string;
    callback: CallbackT;
    shutdown?: ToolShutdown;
    /**
     * Whether or not information about this tool is safe to send to telemetry without obfuscation.
     * - If `true`/`false`, then it will be assumed that all such information is safe/unsafe.
     * - If an object, then safety is determined per property.
     */
    safeForTelemetry?: {
        name: boolean;
        inputsNames: boolean;
    } | true;
};

/**
 * @param input The input to the tool
 * @param options Options for the tool, includes the standard `ToolCallbackOptions` as well as any additional options that were set for the tool in settings.
 */
declare type ToolCallback = (input: unknown, options?: ToolCallbackOptions) => Promise<ToolResult>;

declare type ToolCallbackOptions<OptionsT = {
    [key: string]: unknown;
}> = {
    /**
     * The ID of the LLM tool call which initiated this tool invocation.
     */
    toolCallId: string;
    truncationOptions?: {
        /**
         * The number of tokens that the tool's response should ideally be limited to.
         */
        tokenLimit: number;
        /**
         * A function to count the number of tokens in a string.
         */
        countTokens: (input: string) => number;
    };
    /**
     * A client that the tool can use to make chat completion calls.
     */
    client?: Client_2;
    /**
     * Other options specific to the tool. Passed in from settings.
     */
    toolOptions?: OptionsT;
    /**
     * Global runtime settings.
     */
    settings: RuntimeSettings;
    /**
     * An optional AbortSignal to allow cancellation of tool execution.
     */
    abortSignal?: AbortSignal;
    /**
     * Options for handling large tool outputs.
     */
    largeOutputOptions?: LargeOutputOptions;
};

export declare type ToolExecutionCompleteEvent = z.infer<typeof ToolExecutionCompleteEventSchema>;

/**
 * Tool execution completes (success or error)
 */
declare const ToolExecutionCompleteEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"tool.execution_complete">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        success: z.ZodBoolean;
        isUserRequested: z.ZodOptional<z.ZodBoolean>;
        result: z.ZodOptional<z.ZodObject<{
            /**
             * Tool result content sent to LLM for chat completion.
             * Typically concise/truncated for token efficiency.
             * Populated from: textResultForLlm || sessionLog
             */
            content: z.ZodString;
            /**
             * Detailed tool result for UI/timeline display.
             * Preserves full content like diffs. Optional - falls back to content.
             * Populated from: sessionLog || textResultForLlm
             */
            detailedContent: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            content: string;
            detailedContent?: string | undefined;
        }, {
            content: string;
            detailedContent?: string | undefined;
        }>>;
        error: z.ZodOptional<z.ZodObject<{
            message: z.ZodString;
            code: z.ZodOptional<z.ZodString>;
        }, "strip", z.ZodTypeAny, {
            message: string;
            code?: string | undefined;
        }, {
            message: string;
            code?: string | undefined;
        }>>;
        toolTelemetry: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodUnknown>>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    }, {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    };
    id: string;
    type: "tool.execution_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        success: boolean;
        result?: {
            content: string;
            detailedContent?: string | undefined;
        } | undefined;
        error?: {
            message: string;
            code?: string | undefined;
        } | undefined;
        toolTelemetry?: Record<string, unknown> | undefined;
        parentToolCallId?: string | undefined;
        isUserRequested?: boolean | undefined;
    };
    id: string;
    type: "tool.execution_complete";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

declare type ToolExecutionEvent = {
    kind: "tool_execution";
    turn: number;
    callId?: string;
    toolCallId: string;
    toolResult: ToolResultExpanded;
    durationMs: number;
};

export declare type ToolExecutionPartialResultEvent = z.infer<typeof ToolExecutionPartialResultEventSchema>;

/**
 * Tool execution partial result (streaming updates)
 * Note: These events have ephemeral: true and are NOT persisted to disk
 */
declare const ToolExecutionPartialResultEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"tool.execution_partial_result">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        partialOutput: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        partialOutput: string;
    }, {
        toolCallId: string;
        partialOutput: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        partialOutput: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_partial_result";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        toolCallId: string;
        partialOutput: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_partial_result";
    timestamp: string;
    parentId: string | null;
}>;

export declare type ToolExecutionProgressEvent = z.infer<typeof ToolExecutionProgressEventSchema>;

/**
 * Tool execution progress notification (e.g., from MCP servers)
 * Note: These events have ephemeral: true and are NOT persisted to disk
 */
declare const ToolExecutionProgressEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
} & {
    type: z.ZodLiteral<"tool.execution_progress">;
    ephemeral: z.ZodLiteral<true>;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        progressMessage: z.ZodString;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        progressMessage: string;
    }, {
        toolCallId: string;
        progressMessage: string;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        progressMessage: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_progress";
    timestamp: string;
    parentId: string | null;
}, {
    data: {
        toolCallId: string;
        progressMessage: string;
    };
    id: string;
    ephemeral: true;
    type: "tool.execution_progress";
    timestamp: string;
    parentId: string | null;
}>;

export declare type ToolExecutionStartEvent = z.infer<typeof ToolExecutionStartEventSchema>;

/**
 * Tool execution begins
 */
declare const ToolExecutionStartEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"tool.execution_start">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        toolName: z.ZodString;
        arguments: z.ZodUnknown;
        mcpServerName: z.ZodOptional<z.ZodString>;
        mcpToolName: z.ZodOptional<z.ZodString>;
    } & {
        parentToolCallId: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    }, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "tool.execution_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
        mcpServerName?: string | undefined;
        mcpToolName?: string | undefined;
        parentToolCallId?: string | undefined;
    };
    id: string;
    type: "tool.execution_start";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

/**
 * JSON Schema object for tool input parameters.
 *
 * Tool input schemas are always objects (type: "object") since function calling
 * requires named parameters. This type is compatible with:
 * - OpenAI's `FunctionParameters` (`{ [key: string]: unknown }`)
 * - Anthropic's `Tool.InputSchema` (`{ type: 'object'; [k: string]: unknown }`)
 *
 * When accessing specific schema properties (e.g., `properties`, `required`),
 * explicit type narrowing or assertions are required.
 */
declare type ToolInputSchema = {
    type: "object";
    [key: string]: unknown;
};

/**
 * An event that is emitted by the `Client` for each tool message it will send back to the LLM.
 */
declare type ToolMessageEvent = {
    kind: "message";
    turn?: number;
    callId?: string;
    modelCall?: ModelCallParam;
    message: ChatCompletionToolMessageParam;
};

/**
 * Lightweight tool metadata used for system message building, token counting, and display.
 * Contains only the properties needed for these purposes, without execution callbacks.
 *
 * This is the base type that Tool extends. Functions that only need tool metadata
 * (like cliSystemMessage) should accept ToolMetadata[] to allow passing either
 * ToolMetadata[] or Tool[] without conversion.
 */
declare type ToolMetadata = {
    /**
     * Name used to identify the tool in prompts and tool calls.
     */
    name: string;
    /**
     * Optional namespaced name for the tool used for declarative filtering of tools.
     * e.g.: "playwright/navigate"
     */
    namespacedName?: string;
    /**
     * Optional MCP metadata.
     * These are only set for MCP-backed tools and are used for telemetry/logging.
     *
     * Note: `mcpServerName` is a display/original name and may contain "/".
     */
    mcpServerName?: string;
    mcpToolName?: string;
    /** Intended for UI and end-user contexts — optimized to be human-readable
     * and easily understood, even by those unfamiliar with domain-specific terminology.
     *
     * If not provided, the name should be used for display (except for Tool, where
     * annotations.title should be given precedence over using name, if present).
     */
    title?: string;
    /**
     * Description of what the tool does.
     */
    description: string;
    /**
     * JSON Schema for the tool's input.
     * Required for function tools, optional for custom tools.
     */
    input_schema?: ToolInputSchema;
    /**
     * Optional instructions for how to use this tool effectively.
     * These instructions will be included in the system prompt's <tools> section.
     */
    instructions?: string;
};

declare type ToolProgressCallback = (callId: string, progressMessage: string) => void;

declare type ToolResult = string | ToolResultExpanded;

declare type ToolResultExpanded<TelemetryT extends Telemetry = Telemetry> = {
    /**
     * The result to be given back to the LLM.
     *
     * If @see sessionLog is omitted, then this will be used as the session log.
     */
    textResultForLlm: string;
    /**
     * The result to be given back to the LLM. It can be either base64 encoded image or audio content.
     */
    binaryResultsForLlm?: BinaryResult[];
    /**
     * Whether or not the result should be considered a success, failure, or previously interrupted.
     * - `success`: The tool executed successfully and produced a valid result.
     * - `failure`: The tool encountered an error or did not produce a valid result.
     * - `rejected`: The tool call was rejected either because the user didn't want this call, or a previous dependent one.
     * - `denied`: The tool call was denied because the permissions service said no.
     */
    resultType: "success" | "failure" | "rejected" | "denied";
    /**
     * If there was any sort of error that caused the tool to fail, then a string representation of the error. Typically
     * only set if {@link resultType} is `'failure'`.
     */
    error?: string;
    /**
     * Specific telemetry for the tool. Will be sent back to the server by the agent.
     */
    toolTelemetry?: {
        properties?: TelemetryT["properties"];
        restrictedProperties?: TelemetryT["restrictedProperties"];
        metrics?: TelemetryT["metrics"];
    };
    /**
     * Well-formatted (typically Markdown) string that can be used to display the input/output of the tool invoked.
     *
     * (Optional) If omitted, the text result for the LLM will be used as the session log.
     */
    sessionLog?: string;
    /**
     * When true, skips the large output processing that would normally write
     * large results to a temp file. Used when the caller explicitly requested
     * the full output (e.g., forceReadLargeFiles=true on the view tool).
     */
    skipLargeOutputProcessing?: boolean;
    /**
     * User messages to inject into the conversation history after this tool result.
     * These messages are added to the conversation and sent to the model, but can be
     * filtered from the timeline display based on their source.
     *
     * Use case: Skills inject their full content as user messages so the model
     * treats them as instructions to follow, while keeping the timeline clean.
     */
    newMessages?: InjectedUserMessage[];
};

/**
 * A callback to be called when the tool is shutting down. Gives the tool
 * a chance to clean things up, and return a telemetry event (if desired) which
 * will be emitted by the agent.
 */
declare type ToolShutdown = () => Promise<TelemetryEvent | void>;

export declare type ToolUserRequestedEvent = z.infer<typeof ToolUserRequestedEventSchema>;

/**
 * Tool user requested event
 */
declare const ToolUserRequestedEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"tool.user_requested">;
    data: z.ZodObject<{
        toolCallId: z.ZodString;
        toolName: z.ZodString;
        arguments: z.ZodUnknown;
    }, "strip", z.ZodTypeAny, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    }, {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    };
    id: string;
    type: "tool.user_requested";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        toolCallId: string;
        toolName: string;
        arguments?: unknown;
    };
    id: string;
    type: "tool.user_requested";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

/**
 * Converts full Tool objects to lightweight ToolMetadata.
 * Used to create a safe-to-cache representation without callbacks.
 */
export declare function toToolMetadata(tools: Tool_2[]): ToolMetadata[];

declare interface TransportFactory {
    createTransport(transportOptions: TransportOptions): StdioClientTransport | StreamableHTTPClientTransport | SSEClientTransport | InMemoryClientTransport;
}

declare type TransportOptions = StdioTransportConfig | HTTPTransportConfig | SSETransportConfig | InMemoryTransportConfig;

declare type TruncationEvent = {
    kind: "history_truncated";
    turn: number;
    performedBy: string;
    truncateResult: {
        tokenLimit: number;
        preTruncationTokensInMessages: number;
        preTruncationMessagesLength: number;
        postTruncationTokensInMessages: number;
        postTruncationMessagesLength: number;
        tokensRemovedDuringTruncation: number;
        messagesRemovedDuringTruncation: number;
    };
};

declare type TurnEvent = {
    kind: "turn_started" | "turn_ended" | "turn_failed" | "turn_retry";
    model: string;
    modelInfo: object;
    turn: number;
    timestampMs: number;
    error?: string;
};

export declare type UpdatableSessionOptions = Omit<SessionOptions, "sessionId" | "startTime" | "modifiedTime" | "summary">;

/**
 * A permission request for accessing URLs.
 */
declare type UrlPermissionRequest = {
    readonly kind: "url";
    /** The intention, e.g. "Fetch web content" */
    readonly intention: string;
    /** The URL being accessed */
    readonly url: string;
};

/**
 * Event emitted every turn to report current context window token usage.
 * Unlike TruncationEvent, this is always emitted regardless of whether truncation occurred.
 */
declare type UsageInfoEvent = {
    kind: "usage_info";
    turn: number;
    tokenLimit: number;
    currentTokens: number;
    messagesLength: number;
};

/** Represents the user-based authentication information (OAuth). */
declare type UserAuthInfo = {
    readonly type: "user";
    readonly host: string;
    readonly login: string;
};

export declare type UserMessageEvent = z.infer<typeof UserMessageEventSchema>;

/**
 * An event that is emitted by the `Client` for each user message it adds to the middle of the conversation.
 */
declare type UserMessageEvent_2 = {
    kind: "message";
    turn?: number;
    callId?: string;
    modelCall?: ModelCallParam;
    message: ChatCompletionUserMessageParam;
    /**
     * The component which was the source of the user message.
     * - `jit-instruction`: The message was injected by something which adds automated instructions for the agent
     * - `command-{id}`: The message was injected as a result of a command with the given id
     * - `string`: Some other source
     */
    source?: string;
};

/**
 * User message with optional attachments
 */
declare const UserMessageEventSchema: z.ZodObject<{
    id: z.ZodString;
    timestamp: z.ZodString;
    parentId: z.ZodNullable<z.ZodString>;
    ephemeral: z.ZodOptional<z.ZodBoolean>;
} & {
    type: z.ZodLiteral<"user.message">;
    data: z.ZodObject<{
        content: z.ZodString;
        transformedContent: z.ZodOptional<z.ZodString>;
        attachments: z.ZodOptional<z.ZodArray<z.ZodDiscriminatedUnion<"type", [z.ZodObject<{
            path: z.ZodString;
            displayName: z.ZodString;
        } & {
            type: z.ZodLiteral<"file">;
        }, "strip", z.ZodTypeAny, {
            type: "file";
            path: string;
            displayName: string;
        }, {
            type: "file";
            path: string;
            displayName: string;
        }>, z.ZodObject<{
            path: z.ZodString;
            displayName: z.ZodString;
        } & {
            type: z.ZodLiteral<"directory">;
        }, "strip", z.ZodTypeAny, {
            type: "directory";
            path: string;
            displayName: string;
        }, {
            type: "directory";
            path: string;
            displayName: string;
        }>, z.ZodObject<{
            type: z.ZodLiteral<"selection">;
            filePath: z.ZodString;
            displayName: z.ZodString;
            text: z.ZodString;
            selection: z.ZodObject<{
                start: z.ZodObject<{
                    line: z.ZodNumber;
                    character: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    line: number;
                    character: number;
                }, {
                    line: number;
                    character: number;
                }>;
                end: z.ZodObject<{
                    line: z.ZodNumber;
                    character: z.ZodNumber;
                }, "strip", z.ZodTypeAny, {
                    line: number;
                    character: number;
                }, {
                    line: number;
                    character: number;
                }>;
            }, "strip", z.ZodTypeAny, {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            }, {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            }>;
        }, "strip", z.ZodTypeAny, {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        }, {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        }>]>, "many">>;
        source: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    }, {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    }>;
}, "strip", z.ZodTypeAny, {
    data: {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    };
    id: string;
    type: "user.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}, {
    data: {
        content: string;
        source?: string | undefined;
        transformedContent?: string | undefined;
        attachments?: ({
            type: "file";
            path: string;
            displayName: string;
        } | {
            type: "directory";
            path: string;
            displayName: string;
        } | {
            type: "selection";
            text: string;
            filePath: string;
            displayName: string;
            selection: {
                end: {
                    line: number;
                    character: number;
                };
                start: {
                    line: number;
                    character: number;
                };
            };
        })[] | undefined;
    };
    id: string;
    type: "user.message";
    timestamp: string;
    parentId: string | null;
    ephemeral?: boolean | undefined;
}>;

export declare type UserPromptSubmittedHook = (input: UserPromptSubmittedHookInput) => Promise<UserPromptSubmittedHookOutput | void>;

/**
 * User prompt submitted hook types
 */
export declare interface UserPromptSubmittedHookInput extends BaseHookInput {
    prompt: string;
}

export declare interface UserPromptSubmittedHookOutput {
    modifiedPrompt?: string;
    additionalContext?: string;
    suppressOutput?: boolean;
}

declare type WildcardEventHandler = (event: SessionEvent) => void;

/**
 * Working directory context for session tracking
 */
declare interface WorkingDirectoryContext {
    /** Current working directory */
    cwd: string;
    /** Git repository root, if in a git repo */
    gitRoot?: string;
    /** GitHub repository in "owner/repo" format, if available */
    repository?: string;
    /** Current git branch, if in a git repo */
    branch?: string;
}

declare type Workspace = z_2.infer<typeof WorkspaceSchema>;

/**
 * Working directory context for workspace creation/update
 */
declare interface WorkspaceContext {
    cwd?: string;
    gitRoot?: string;
    repository?: string;
    branch?: string;
}

/**
 * Information about the current workspace for context injection into prompts.
 */
declare interface WorkspaceContextInfo {
    /** Name of the workspace (for display) */
    name?: string;
    /** Path to the workspace directory */
    workspacePath: string;
    /** Number of prior session summaries */
    summaryCount: number;
    /** Whether a plan.md file exists */
    hasPlan: boolean;
    /** List of files in the files/ directory (if any) */
    filesInWorkspace?: string[];
    /** List of checkpoints with their titles */
    checkpoints?: CheckpointInfo[];
}

/**
 * Schema for workspace metadata.
 * Stored at: ~/.copilot/session-state/{session-id}/workspace.yaml
 */
declare const WorkspaceSchema: z_2.ZodObject<{
    id: z_2.ZodString;
    cwd: z_2.ZodOptional<z_2.ZodString>;
    git_root: z_2.ZodOptional<z_2.ZodString>;
    repository: z_2.ZodOptional<z_2.ZodString>;
    branch: z_2.ZodOptional<z_2.ZodString>;
    summary: z_2.ZodOptional<z_2.ZodString>;
    name: z_2.ZodOptional<z_2.ZodString>;
    summary_count: z_2.ZodDefault<z_2.ZodNumber>;
    created_at: z_2.ZodOptional<z_2.ZodString>;
    updated_at: z_2.ZodOptional<z_2.ZodString>;
}, "strip", z_2.ZodTypeAny, {
    id: string;
    summary_count: number;
    name?: string | undefined;
    branch?: string | undefined;
    summary?: string | undefined;
    cwd?: string | undefined;
    repository?: string | undefined;
    git_root?: string | undefined;
    created_at?: string | undefined;
    updated_at?: string | undefined;
}, {
    id: string;
    name?: string | undefined;
    branch?: string | undefined;
    summary?: string | undefined;
    cwd?: string | undefined;
    repository?: string | undefined;
    git_root?: string | undefined;
    summary_count?: number | undefined;
    created_at?: string | undefined;
    updated_at?: string | undefined;
}>;

/**
 * A permission request for writing to new or existing files.
 */
declare type WritePermissionRequest = {
    readonly kind: "write";
    /** The intention of the edit operation, e.g. "Edit file" or "Create file" */
    readonly intention: string;
    /** The name of the file being edited */
    readonly fileName: string;
    /** The diff of the changes being made */
    readonly diff: string;
    /** The new file contents (for IDE diff view) */
    readonly newFileContents?: string;
};

export { }
