/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc.array;

import gov.nasa.giss.data.nc.NcArray;
import gov.nasa.giss.data.nc.NcDataset;
import gov.nasa.giss.data.nc.NcDimension;
import gov.nasa.giss.data.nc.NcException;
import gov.nasa.giss.data.nc.NcUtils;
import gov.nasa.giss.data.nc.NcVariable;
import gov.nasa.giss.data.nc.array.NcArrayLonLat;
import gov.nasa.giss.data.nc.array.NcArrayLonLatUgrid;
import gov.nasa.giss.data.nc.array.NcArrayLonLatUgridUtils;
import gov.nasa.giss.data.nc.gridder.NcGridder;
import gov.nasa.giss.data.nc.gridder.NcGridderLonLatUgrid;
import gov.nasa.giss.math.GeometryUtils;
import gov.nasa.giss.text.PrintfFormat;
import java.lang.invoke.MethodHandles;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.ma2.ArrayInt;
import ucar.nc2.Dimension;
import ucar.nc2.dataset.VariableDS;

public class NcArrayLonLatUgridEdges
extends NcArray
implements NcArrayLonLat,
NcArrayLonLatUgrid,
NcArrayLonLatUgridUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final double RAD_PER_DEG = Math.PI / 180;
    private String meshStr_;
    private int numEdges_ = -1;
    private int edgeDimIndex_ = -1;
    private VariableDS edgeLonVarDS_;
    private VariableDS edgeLatVarDS_;
    private double[] edgeLons_;
    private double[] edgeLats_;
    private double[][] edgeBoundLons_;
    private double[][] edgeBoundLats_;
    private int[][] edgeFaces_;
    private int[][] edgeNodes_;
    private int[][] closestEdges_;

    public NcArrayLonLatUgridEdges(NcVariable ncvar) throws NcException {
        super(ncvar);
        if (!NcArrayLonLatUgridUtils.hasUgridAttributes(this.dataset_, this.njvarDS_, "edge")) {
            throw new NcException("Variable does not have valid UGRID attributes");
        }
        this.meshStr_ = NcArrayLonLatUgridUtils.getMeshName(this.njvarDS_);
        if (this.meshStr_ == null) {
            throw new NcException("Variable has no mesh attribute");
        }
        this.createCoordinates();
        this.initGridArrays();
        this.initGridBounds();
    }

    private void createCoordinates() {
        VariableDS[] cvars = NcArrayLonLatUgridUtils.getCoordinateVars(this.ncvar_);
        if (cvars == null) {
            throw new NcException("Coordinate vars not found.");
        }
        this.edgeLonVarDS_ = cvars[0];
        this.edgeLatVarDS_ = cvars[1];
        String lonName = this.edgeLonVarDS_.getShortName();
        String latName = this.edgeLatVarDS_.getShortName();
        LOGGER.trace("== Lon-Lat CVs named {}, {}", (Object)lonName, (Object)latName);
        if (!lonName.startsWith(this.meshStr_)) {
            throw new NcException("Longitude CV name does not start with '" + this.meshStr_ + "'");
        }
        if (!latName.startsWith(this.meshStr_)) {
            throw new NcException("Latitude CV name does not start with '" + this.meshStr_ + "'");
        }
        if (!lonName.endsWith("_edge_x")) {
            throw new NcException("Longitude CV name does not end in 'edge_x'");
        }
        if (!latName.endsWith("_edge_y")) {
            throw new NcException("Latitude CV name does not end in 'edge_y'");
        }
        if (this.edgeLonVarDS_.getRank() != 1 || this.edgeLatVarDS_.getRank() != 1) {
            throw new NcException("Lon and lat coordinate variable(s) are not 1D");
        }
        Dimension edgedim = this.edgeLonVarDS_.getDimension(0);
        LOGGER.trace("edgedim {}", (Object)edgedim);
        this.edgeDimIndex_ = this.findDimensionIndex(this.edgeLonVarDS_);
        LOGGER.trace("edgeDimIndex {}", (Object)this.edgeDimIndex_);
        if (this.edgeDimIndex_ < 0) {
            throw new NcException("Face dimension index is not set.");
        }
        this.numEdges_ = edgedim.getLength();
        LOGGER.trace("numEdges, {}", (Object)this.numEdges_);
    }

    private void initGridArrays() {
        Array latArray;
        Array lonArray;
        try {
            lonArray = this.edgeLonVarDS_.read();
        }
        catch (Exception exc) {
            LOGGER.warn("Could not read longitude coordinates array: {}", (Object)exc.getMessage());
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException("Could not read longitude coordinates array.", exc);
        }
        try {
            latArray = this.edgeLatVarDS_.read();
        }
        catch (Exception exc) {
            LOGGER.debug("Could not read latitude coordinates array: {}", (Object)exc.getMessage());
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException("Could not read latitude coordinates array.", exc);
        }
        this.edgeLons_ = new double[this.numEdges_];
        this.edgeLats_ = new double[this.numEdges_];
        for (int i = 0; i < this.numEdges_; ++i) {
            this.edgeLons_[i] = lonArray.getDouble(i);
            this.edgeLats_[i] = latArray.getDouble(i);
        }
    }

    private void initGridBounds() {
        int i;
        LOGGER.trace("Extracting edge node ints");
        VariableDS edgeNodesVar = NcArrayLonLatUgridUtils.getEdgeNodesVar(this.dataset_, this.njvarDS_);
        ArrayInt.D2 edgeNodesArray = NcArrayLonLatUgridUtils.getArrayInt2D(edgeNodesVar, 2);
        if (edgeNodesArray.getShape()[0] != this.numEdges_) {
            throw new NcException("Edge nodes length does not match edge count.");
        }
        Integer enstart = NcUtils.getAttributeIntegerIgnoreCase(edgeNodesVar, "start_index");
        LOGGER.trace("EN start_index, {}", (Object)enstart);
        if (enstart == null) {
            enstart = 0;
        }
        if (enstart < 0 || enstart > 1) {
            LOGGER.trace("Edge nodes variable has strange start_index value, {}", (Object)enstart);
            throw new NcException("Edge nodes variable trange start_index value.");
        }
        this.edgeNodes_ = new int[this.numEdges_][2];
        for (int i2 = 0; i2 < this.numEdges_; ++i2) {
            this.edgeNodes_[i2][0] = edgeNodesArray.get(i2, 0) - enstart;
            this.edgeNodes_[i2][1] = edgeNodesArray.get(i2, 1) - enstart;
        }
        LOGGER.trace("Extracting edge face ints");
        VariableDS edgeFacesVar = NcArrayLonLatUgridUtils.getEdgeFacesVar(this.dataset_, this.njvarDS_);
        ArrayInt.D2 edgeFacesArray = NcArrayLonLatUgridUtils.getArrayInt2D(edgeFacesVar, 2);
        if (edgeFacesArray.getShape()[0] != this.numEdges_) {
            throw new NcException("Edge faces length does not match edge count.");
        }
        Integer efstart = NcUtils.getAttributeIntegerIgnoreCase(edgeFacesVar, "start_index");
        LOGGER.trace("EF start_index, {}", (Object)efstart);
        if (efstart == null) {
            efstart = 0;
        }
        if (efstart < 0 || efstart > 1) {
            LOGGER.trace("Edge faces variable has strange start_index value, {}", (Object)efstart);
            throw new NcException("Edge faces variable trange start_index value.");
        }
        this.edgeFaces_ = new int[this.numEdges_][2];
        for (i = 0; i < this.numEdges_; ++i) {
            this.edgeFaces_[i][0] = edgeFacesArray.get(i, 0) - efstart;
            this.edgeFaces_[i][1] = edgeFacesArray.get(i, 1) - efstart;
        }
        this.edgeBoundLons_ = new double[this.numEdges_][4];
        this.edgeBoundLats_ = new double[this.numEdges_][4];
        for (i = 0; i < this.numEdges_; ++i) {
            for (int j = 0; j < 4; ++j) {
                this.edgeBoundLons_[i][j] = Double.NaN;
                this.edgeBoundLats_[i][j] = Double.NaN;
            }
        }
        VariableDS[] nodeVars = NcArrayLonLatUgridUtils.getNodeCoordinateVars(this.dataset_, this.njvarDS_);
        if (nodeVars == null) {
            throw new NcException("Cannot get edge coordinate bounds because there are no node vars");
        }
        Array nodeLonArray = null;
        Array nodeLatArray = null;
        try {
            nodeLonArray = nodeVars[0].read();
            nodeLatArray = nodeVars[1].read();
        }
        catch (Exception exc) {
            LOGGER.trace("Exception reading node lon or lar coordinates Array, {}", (Object)exc.getMessage());
            throw new NcException("Failed to read node coordinates Array.");
        }
        for (int i3 = 0; i3 < this.numEdges_; ++i3) {
            int node1 = this.edgeNodes_[i3][0];
            int node2 = this.edgeNodes_[i3][1];
            this.edgeBoundLons_[i3][0] = nodeLonArray.getDouble(node1);
            this.edgeBoundLats_[i3][0] = nodeLatArray.getDouble(node1);
            this.edgeBoundLons_[i3][2] = nodeLonArray.getDouble(node2);
            this.edgeBoundLats_[i3][2] = nodeLatArray.getDouble(node2);
        }
        VariableDS[] faceVars = NcArrayLonLatUgridUtils.getFaceCoordinateVars(this.dataset_, this.njvarDS_);
        if (faceVars == null) {
            throw new NcException("Cannot get edge coordinate bounds because there are no face vars");
        }
        Array faceLonArray = null;
        Array faceLatArray = null;
        try {
            faceLonArray = faceVars[0].read();
            faceLatArray = faceVars[1].read();
        }
        catch (Exception exc) {
            LOGGER.trace("Exception reading face lon or lar coordinates Array, {}", (Object)exc.getMessage());
            throw new NcException("Failed to create face coordinates Array.");
        }
        for (int i4 = 0; i4 < this.numEdges_; ++i4) {
            int face1 = this.edgeFaces_[i4][0];
            int face2 = this.edgeFaces_[i4][1];
            if (face1 < 0 && face2 > face1) {
                face1 = face2;
                face2 = -1;
            }
            if (face1 > -1) {
                this.edgeBoundLons_[i4][1] = faceLonArray.getDouble(face1);
                this.edgeBoundLats_[i4][1] = faceLatArray.getDouble(face1);
            }
            if (face2 <= -1) continue;
            this.edgeBoundLons_[i4][3] = faceLonArray.getDouble(face2);
            this.edgeBoundLats_[i4][3] = faceLatArray.getDouble(face2);
        }
    }

    private void findClosestEdgesUnused() {
        int i;
        this.closestEdges_ = new int[this.numEdges_][4];
        for (i = 0; i < this.numEdges_; ++i) {
            for (int j = 0; j < 4; ++j) {
                this.closestEdges_[i][j] = -1;
            }
        }
        LOGGER.trace("Extracting closest edges");
        for (i = 0; i < this.numEdges_; ++i) {
            if (this.closestEdges_[i][3] >= 0) continue;
            block3: for (int k = i + 1; k < this.numEdges_; ++k) {
                int j;
                if (this.closestEdges_[k][3] >= 0 || this.edgeFaces_[k][0] != this.edgeFaces_[i][0] && this.edgeFaces_[k][0] != this.edgeFaces_[i][1] && this.edgeFaces_[k][1] != this.edgeFaces_[i][0] && this.edgeFaces_[k][1] != this.edgeFaces_[i][1]) continue;
                for (j = 0; j < 4; ++j) {
                    if (this.closestEdges_[i][j] >= 0) continue;
                    this.closestEdges_[i][j] = k;
                    break;
                }
                for (j = 0; j < 4; ++j) {
                    if (this.closestEdges_[k][j] >= 0) continue;
                    this.closestEdges_[k][j] = i;
                    continue block3;
                }
            }
        }
    }

    @Override
    public int getLength() {
        return this.numEdges_;
    }

    @Override
    public double longitudeAt(int index) {
        return this.edgeLons_[index];
    }

    @Override
    public double latitudeAt(int index) {
        return this.edgeLats_[index];
    }

    @Override
    public double valueAt(int index) {
        if (index < 0 || index >= this.numEdges_) {
            throw new IndexOutOfBoundsException("Index out of range: " + index + ", " + this.numEdges_);
        }
        if (this.needsSlice_) {
            this.doSlice();
        }
        try {
            return this.getDoubleFromSlice(index);
        }
        catch (Exception exc) {
            LOGGER.error("Slice getDouble failed -  index = {}.{}", (Object)index);
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException(exc.toString());
        }
    }

    @Override
    public double[][] cornerLonLats(int index) {
        double lat0;
        double lon0 = this.longitudeAt(index);
        if (NcArrayLonLat.isBadLonLat(lon0, lat0 = this.latitudeAt(index))) {
            return null;
        }
        if (lon0 > 180.0) {
            lon0 -= 360.0;
        }
        if (lon0 < -180.0) {
            lon0 += 360.0;
        }
        double lon1 = this.edgeBoundLons_[index][0];
        double lon2 = this.edgeBoundLons_[index][1];
        double lon3 = this.edgeBoundLons_[index][2];
        double lon4 = this.edgeBoundLons_[index][3];
        if (lon0 - lon1 > 45.0) {
            lon1 += 360.0;
        }
        if (lon0 - lon2 > 45.0) {
            lon2 += 360.0;
        }
        if (lon0 - lon3 > 45.0) {
            lon3 += 360.0;
        }
        if (!Double.isNaN(lon4) && lon0 - lon4 > 45.0) {
            lon4 += 360.0;
        }
        if (lon1 - lon0 > 45.0) {
            lon1 -= 360.0;
        }
        if (lon2 - lon0 > 45.0) {
            lon2 -= 360.0;
        }
        if (lon3 - lon0 > 45.0) {
            lon3 -= 360.0;
        }
        if (!Double.isNaN(lon4) && lon4 - lon0 > 45.0) {
            lon4 -= 360.0;
        }
        double lat1 = this.edgeBoundLats_[index][0];
        double lat2 = this.edgeBoundLats_[index][1];
        double lat3 = this.edgeBoundLats_[index][2];
        double lat4 = this.edgeBoundLats_[index][3];
        if (Double.isNaN(lon4)) {
            return new double[][]{{lon1, lat1}, {lon2, lat2}, {lon3, lat3}};
        }
        return new double[][]{{lon1, lat1}, {lon2, lat2}, {lon3, lat3}, {lon4, lat4}};
    }

    @Override
    protected void createFreeDimensions() {
        this.dimensions_ = new NcDimension[this.rank_];
        for (int i = 0; i < this.rank_; ++i) {
            this.dimensions_[i] = i == this.edgeDimIndex_ ? null : this.ncvar_.getDimension(i);
        }
    }

    @Override
    protected void findExtremaImpl() {
        double[] range = this.getActualRangeAttrValues();
        if (range != null) {
            this.minimum_ = range[0];
            this.maximum_ = range[1];
            this.needsExtrema_ = false;
            return;
        }
        for (int i = 0; i < this.numEdges_; ++i) {
            this.findExtremaTestValue(this.valueAt(i));
        }
    }

    @Override
    public NcGridder getGridder() {
        return new NcGridderLonLatUgrid();
    }

    @Override
    public void describeCell(StringBuilder sb, PrintfFormat valFormat, int ... index) {
        Objects.requireNonNull(index, "Edge index cannot be null.");
        int face = index[0];
        if (face < 0 || face >= this.numEdges_) {
            sb.append("Edge index outside valid range.");
            return;
        }
        double lon = this.edgeLons_[face];
        double lat = this.edgeLats_[face];
        double gv = this.valueAt(face);
        sb.append("Edge [").append(face + 1).append(']').append(" at [").append(NcUtils.formatLongitude(lon)).append(' ').append(NcUtils.formatLatitude(lat)).append("], value = ").append(valFormat.sprintfx(gv));
    }

    @Override
    public int[] getEnclosingCell(double lon, double lat) {
        double mindist = Double.POSITIVE_INFINITY;
        double cosLat = Math.cos(lat * (Math.PI / 180));
        double sinLat = Math.sin(lat * (Math.PI / 180));
        int index = -1;
        for (int i = 0; i < this.numEdges_; ++i) {
            double lat1;
            double lon1;
            try {
                lon1 = this.longitudeAt(i);
                lat1 = this.latitudeAt(i);
            }
            catch (Exception exc) {
                continue;
            }
            if (Double.isNaN(lon1) || Double.isNaN(lat1)) continue;
            double dlon = lon1 - lon;
            double cosLat1 = Math.cos(lat1 * (Math.PI / 180));
            double sinLat1 = Math.sin(lat1 * (Math.PI / 180));
            double cosAngle = cosLat * cosLat1 * Math.cos(dlon * (Math.PI / 180)) + sinLat * sinLat1;
            double angle = Math.acos(cosAngle);
            if (angle == 0.0) {
                return new int[]{i};
            }
            if (!(angle < mindist)) continue;
            mindist = angle;
            index = i;
        }
        if (index < 0) {
            return null;
        }
        double[][] corners = this.cornerLonLats(index);
        if (corners == null) {
            return null;
        }
        if (corners.length == 4 && GeometryUtils.isPointInQuadrilateral(lon, lat, corners[0][0], corners[0][1], corners[1][0], corners[1][1], corners[2][0], corners[2][1], corners[3][0], corners[3][1])) {
            return new int[]{index};
        }
        if (corners.length == 3 && GeometryUtils.isPointInTriangle(lon, lat, corners[0][0], corners[0][1], corners[1][0], corners[1][1], corners[2][0], corners[2][1])) {
            return new int[]{index};
        }
        return null;
    }

    private double getDoubleFromSlice(int index) {
        int[] ss = new int[this.rank_];
        for (int i = 0; i < this.rank_; ++i) {
            ss[i] = 0;
        }
        ss[this.edgeDimIndex_] = index;
        this.sliceIdx_.set(ss);
        return this.slice_.getDouble(this.sliceIdx_);
    }

    private synchronized void doSlice() throws NcException {
        LOGGER.trace("");
        int[] sOrigin = new int[this.rank_];
        int[] sShape = new int[this.rank_];
        try {
            this.needsSlice_ = true;
            for (int i = 0; i < this.rank_; ++i) {
                sOrigin[i] = this.sIndex_[i];
                sShape[i] = 1;
            }
            sOrigin[this.edgeDimIndex_] = 0;
            sShape[this.edgeDimIndex_] = this.numEdges_;
            this.slice_ = this.njvarDS_.read(sOrigin, sShape);
            this.sliceIdx_ = this.slice_.getIndex();
            this.needsSlice_ = false;
        }
        catch (Exception exc) {
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            throw new NcException("Do Slice - " + exc.toString());
        }
    }

    public static boolean canGridVariable(NcVariable ncvar) {
        NcDataset ncd = ncvar.getDataset();
        VariableDS varDS = (VariableDS)ncvar.getObject();
        return NcArrayLonLatUgridEdges.canGridVariable(ncd, varDS);
    }

    public static boolean canGridVariable(NcDataset ncd, VariableDS varDS) {
        LOGGER.trace("");
        if (!NcArrayLonLatUgridUtils.hasUgridAttributes(ncd, varDS, "edge")) {
            return false;
        }
        VariableDS[] gridvars = NcArrayLonLatUgridUtils.getCoordinateVars(ncd, varDS);
        boolean hasGridvars = gridvars != null;
        LOGGER.trace("hasGridvars {}", (Object)hasGridvars);
        if (!hasGridvars) {
            return false;
        }
        return NcArrayLonLatUgridEdges.validateCoordinateVars(ncd, varDS, gridvars[0], gridvars[1]);
    }

    private static boolean validateCoordinateVars(NcDataset ncd, VariableDS varDS, VariableDS lonCV, VariableDS latCV) {
        Dimension dd2;
        if (lonCV == null) {
            LOGGER.trace("Longitude CV is null");
            return false;
        }
        if (latCV == null) {
            LOGGER.trace("Latitude CV is null");
            return false;
        }
        String meshStr = NcArrayLonLatUgridUtils.getMeshName(varDS);
        String lonName = lonCV.getShortName();
        String latName = latCV.getShortName();
        LOGGER.trace("== Lon-Lat CVs named {}, {}", (Object)lonName, (Object)latName);
        if (!lonName.startsWith(meshStr)) {
            LOGGER.trace("Longitude CV name does not start with '{}'", (Object)meshStr);
            return false;
        }
        if (!latName.startsWith(meshStr)) {
            LOGGER.trace("Latitude CV name does not start with '{}'", (Object)meshStr);
            return false;
        }
        if (!lonName.endsWith("_edge_x")) {
            LOGGER.trace("Longitude CV name does not end in 'edge_x'");
            return false;
        }
        if (!latName.endsWith("_edge_y")) {
            LOGGER.trace("Latitude CV name does not end in 'edge_y'");
            return false;
        }
        int varRank = varDS.getRank();
        int lonRank = lonCV.getRank();
        int latRank = latCV.getRank();
        LOGGER.trace("== ranks: var {}, lon {}, lat {}", varRank, lonRank, latRank);
        if (lonRank != 1) {
            LOGGER.trace("Longitude CV must be rank 1.");
            return false;
        }
        if (latRank != 1) {
            LOGGER.trace("Latitude CV must be rank 1.");
            return false;
        }
        Dimension dd1 = lonCV.getDimension(0);
        if (!dd1.equals(dd2 = latCV.getDimension(0))) {
            return false;
        }
        int didx = NcUtils.findDimensionIndex(varDS, dd1);
        if (didx < 0) {
            return false;
        }
        VariableDS edgeFacesVar = NcArrayLonLatUgridUtils.getEdgeFacesVar(ncd, varDS);
        VariableDS edgeNodesVar = NcArrayLonLatUgridUtils.getEdgeNodesVar(ncd, varDS);
        LOGGER.trace("== has edge_faces, {}", (Object)(edgeFacesVar != null ? 1 : 0));
        LOGGER.trace("== has edge_nodes, {}", (Object)(edgeNodesVar != null ? 1 : 0));
        return edgeFacesVar != null && edgeNodesVar != null;
    }
}

