/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.data.nc;

import gov.nasa.giss.data.nc.NcConfig;
import gov.nasa.giss.data.nc.NcDataGrouping;
import gov.nasa.giss.data.nc.NcDataNode;
import gov.nasa.giss.data.nc.NcException;
import gov.nasa.giss.data.nc.NcFileUtils;
import gov.nasa.giss.data.nc.NcGeometryType;
import gov.nasa.giss.data.nc.NcGroup;
import gov.nasa.giss.data.nc.NcVariable;
import gov.nasa.giss.data.nc.exc.NcForbiddenException;
import gov.nasa.giss.data.nc.exc.NcNotAuthorizedException;
import gov.nasa.giss.data.nc.exc.NcResourceNotFoundException;
import gov.nasa.giss.text.StringUtils;
import gov.nasa.giss.ui.treetable.TreeTableNode;
import java.awt.HeadlessException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.net.URL;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.Attribute;
import ucar.nc2.Dimension;
import ucar.nc2.Group;
import ucar.nc2.NetcdfFile;
import ucar.nc2.NetcdfFiles;
import ucar.nc2.Variable;
import ucar.nc2.dataset.CoordSysBuilder;
import ucar.nc2.dataset.CoordSysBuilderIF;
import ucar.nc2.dataset.DatasetUrl;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dataset.NetcdfDatasets;
import ucar.nc2.dataset.VariableDS;
import ucar.nc2.iosp.IOServiceProvider;

public class NcDataset
extends NcDataGrouping {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int BUFFER_SIZE = 6473600;
    private static final EnumSet<NetcdfDataset.Enhance> ENHANCEMENT;
    private DatasetsStruct datasets_;
    private SourceType stype_;
    private File file_;
    private URL url_;
    private DatasetUrl durl_;
    private String addStr_;
    private String fname_;
    private NcGeometryType geometry_ = NcGeometryType.GRIDDED;

    public NcDataset(File f) throws IOException, FileNotFoundException {
        super(null, null);
        LOGGER.trace("Constructor w/ file, {}", (Object)f);
        if (!f.exists()) {
            LOGGER.trace("Nothing found at File's path.");
            throw new NcResourceNotFoundException("No file found at the specified path.");
        }
        this.ncdataset_ = this;
        this.stype_ = SourceType.FILE;
        this.file_ = f;
        this.url_ = f.toURI().toURL();
        this.addStr_ = this.url_.toString();
        this.fname_ = this.file_.getName();
        LOGGER.trace("fname {}", (Object)this.fname_);
        String lcname = this.file_.getName().toLowerCase();
        if (this.file_.isDirectory()) {
            LOGGER.trace("File object represents a (Zarr?) directory.");
        } else if (lcname.endsWith(".zip")) {
            LOGGER.trace("File object looks like a ZIP archive.");
        } else if (lcname.endsWith(".ncml")) {
            this.durl_ = NcFileUtils.createDatasetUrl(this.addStr_);
        }
        this.acquire();
        this.finishInit();
    }

    public NcDataset(URL url) throws IOException, FileNotFoundException {
        super(null, null);
        LOGGER.trace("Constructor w/ url, {}", (Object)url);
        this.ncdataset_ = this;
        this.stype_ = SourceType.URL;
        this.url_ = url;
        this.addStr_ = this.url_.toString();
        this.fname_ = NcFileUtils.getUrlFname(this.url_);
        this.durl_ = NcFileUtils.createDatasetUrl(this.addStr_);
        LOGGER.trace("url {}", (Object)this.url_);
        LOGGER.trace("fname {}", (Object)this.fname_);
        String lcname = this.fname_.toLowerCase();
        if (lcname.endsWith("catalog.html") || lcname.endsWith("catalog.xml")) {
            throw new IllegalArgumentException("Address looks like a catalog, not a dataset.");
        }
        this.acquire();
        this.finishInit();
    }

    public NcDataset(DatasetUrl durl) throws IOException, FileNotFoundException {
        super(null, null);
        LOGGER.trace("Constructor w/ durl, {}", (Object)durl);
        this.ncdataset_ = this;
        this.stype_ = SourceType.DATASET_URL;
        this.durl_ = durl;
        this.addStr_ = durl.getTrueurl();
        this.fname_ = NcFileUtils.getDurlFname(durl);
        LOGGER.trace("trueurl {}", (Object)this.addStr_);
        LOGGER.trace("fname {}", (Object)this.fname_);
        this.acquire();
        this.finishInit();
    }

    public NcDataset(String address) throws IOException, FileNotFoundException {
        super(null, null);
        LOGGER.trace("Constructor w/ str, {}", (Object)address);
        this.ncdataset_ = this;
        this.stype_ = SourceType.STRING;
        this.addStr_ = address;
        if (address.toLowerCase().startsWith("file:")) {
            try {
                URI uri = new URI(address);
                this.file_ = new File(uri);
            }
            catch (Exception xc) {
                LOGGER.trace("Could not make URI->File from file: path.");
            }
        } else if (address.startsWith("/")) {
            this.file_ = new File(address);
        }
        this.url_ = NcFileUtils.createUrl(this.addStr_);
        this.durl_ = NcFileUtils.createDatasetUrl(this.addStr_);
        if (this.file_ != null) {
            this.fname_ = this.file_.getName();
        } else if (this.url_ != null) {
            this.fname_ = NcFileUtils.getUrlFname(this.url_);
        } else if (this.durl_ != null) {
            this.fname_ = NcFileUtils.getDurlFname(this.durl_);
        } else {
            LOGGER.warn("Could not construct a File, URL or DatasetUrl from address {}", (Object)address);
            throw new NcException("Could not construct a FILE, URL or DatasetUrl from address");
        }
        LOGGER.trace("file  {}", (Object)this.file_);
        LOGGER.trace("url   {}", (Object)this.url_);
        LOGGER.trace("durl  {}", (Object)this.durl_);
        LOGGER.trace("fname {}", (Object)this.fname_);
        this.acquire();
        this.finishInit();
    }

    private void acquire() throws IOException, FileNotFoundException {
        boolean isNcml;
        LOGGER.trace("stype {}, durl {}", (Object)this.stype_, (Object)(this.durl_ != null ? 1 : 0));
        boolean bl = isNcml = this.fname_ != null && this.fname_.toLowerCase().endsWith(".ncml");
        if (isNcml) {
            this.datasets_ = NcDataset.acquireFromNcml(this.durl_);
            return;
        }
        if (this.file_ != null) {
            this.datasets_ = NcDataset.acquireFromFile(this.file_);
            return;
        }
        if (this.durl_ != null || this.url_ != null) {
            this.datasets_ = NcDataset.acquireFromDurl(this.durl_, this.url_);
            return;
        }
        LOGGER.warn("Address of dataset is invalid or not understood.");
        LOGGER.warn("Type: {}, Address: {}", (Object)this.stype_, (Object)this.addStr_);
        throw new NcException("Address of dataset " + this.fname_ + " is invalid or not understood.");
    }

    private static DatasetsStruct acquireFromFile(File file) throws HeadlessException, IOException {
        LOGGER.trace("{}", (Object)file);
        NetcdfFile njfile = null;
        try {
            njfile = NetcdfFiles.open(file.getPath(), 6473600, null);
        }
        catch (FileNotFoundException hexc) {
            LOGGER.warn("FileNotFoundException acquiring {}", (Object)file);
            throw new NcResourceNotFoundException("No dataset found at path " + file.getPath());
        }
        catch (IOException ioe) {
            LOGGER.warn("IOException acquiring {}, {}", (Object)file, (Object)ioe.getMessage());
            if (LOGGER.isTraceEnabled()) {
                ioe.printStackTrace();
            }
            throw new IOException(ioe.getMessage());
        }
        catch (Exception exc) {
            throw new NcException(exc.getMessage());
        }
        DatasetsStruct datasets = NcDataset.enhanceNjfile(njfile);
        return datasets;
    }

    private static DatasetsStruct acquireFromDurl(DatasetUrl durl, URL url) throws IOException {
        NetcdfFile njfile = null;
        String addStr = null;
        if (durl != null) {
            addStr = durl.getTrueurl();
            LOGGER.trace("DatasetUrl {}", (Object)durl);
            LOGGER.trace("TrueURL {}", (Object)addStr);
            LOGGER.trace("Service type {}", (Object)durl.getServiceType());
        } else if (url != null) {
            addStr = url.toString();
            LOGGER.trace("URL {}", (Object)url);
        } else {
            throw new NcException("Cannot acquire remote dataset w/out a URL or DatasetUrl.");
        }
        try {
            njfile = durl != null ? NetcdfDatasets.acquireFile(null, null, durl, 6473600, null, null) : NetcdfFiles.open(url.toString(), 6473600, null);
        }
        catch (HeadlessException hexc) {
            throw new HeadlessException("HeadlessException acquiring remote at " + addStr + ".");
        }
        catch (FileNotFoundException hexc) {
            LOGGER.warn("FileNotFoundException acquiring {}", (Object)addStr);
            throw new NcResourceNotFoundException("No dataset found at that address.");
        }
        catch (IOException ioe) {
            String msg;
            String lcmsg;
            LOGGER.warn("IOException acquiring {}", (Object)addStr);
            LOGGER.warn("Exc {}", (Object)ioe.getMessage());
            if (LOGGER.isTraceEnabled()) {
                ioe.printStackTrace();
            }
            if ((lcmsg = (msg = ioe.getMessage()).toLowerCase()).contains("400")) {
                LOGGER.trace("400 - Bad HTTP request");
                throw new NcException("Server returned a 'HTTP 400 - Bad Request' error.");
            }
            if (lcmsg.contains("401") || lcmsg.toLowerCase().contains("authoriz")) {
                LOGGER.trace("401 - Authorization required");
                throw new NcNotAuthorizedException();
            }
            if (msg.contains("403")) {
                LOGGER.trace("403 - Access forbidden");
                throw new NcForbiddenException("Access to dataset is forbidden.");
            }
            if (msg.contains("404")) {
                LOGGER.trace("404 - Not found");
                throw new NcResourceNotFoundException("No dataset found at that address.");
            }
            if (msg.contains("405")) {
                LOGGER.trace("405 - Method not allowed");
                throw new IOException("Address seems valid but access method does not. Server returned a 405 error.");
            }
            if (lcmsg.contains("not a valid cdm file")) {
                throw new NcException("Address does not seem to be a netCDF or CDM file that I understand.");
            }
            if (lcmsg.contains("ssl")) {
                LOGGER.debug("SSL protocol exception");
                throw new NcException("Server returned a protocol exception (HTTP vs. HTTPS).");
            }
            if (lcmsg.contains("could not find a matching table for grib file")) {
                throw new NcResourceNotFoundException(NcDataset.getGribMessage(msg));
            }
            throw new NcException(msg);
        }
        catch (Exception exc) {
            String msg = exc.getMessage();
            LOGGER.warn("Exception acquiring dataset: {}", (Object)exc.getClass().getSimpleName());
            LOGGER.warn("Exc {}", (Object)exc.getMessage());
            if (msg.contains("401") || msg.contains("authoriz")) {
                LOGGER.trace("Authorization exception");
                throw new NcNotAuthorizedException();
            }
            if (LOGGER.isTraceEnabled()) {
                exc.printStackTrace();
            }
            if (msg.matches(".*Could not find a table for GRIB file.*")) {
                throw new NcResourceNotFoundException(NcDataset.getGribMessage(msg));
            }
            throw new NcException(msg);
        }
        DatasetsStruct datasets = NcDataset.enhanceNjfile(njfile);
        return datasets;
    }

    private static DatasetsStruct enhanceNjfile(NetcdfFile njfile) throws IOException {
        Objects.requireNonNull(njfile, "NetcdfFile is required.");
        NetcdfDataset enhanced = null;
        NetcdfDataset unenhanced = null;
        LOGGER.trace("Enhance...");
        enhanced = NetcdfDatasets.enhance(njfile, ENHANCEMENT, null);
        LOGGER.trace("Unenhance...");
        unenhanced = NetcdfDatasets.enhance(njfile, null, null);
        LOGGER.trace("Got enhanced {}, unenhanced {}", (Object)(enhanced != null ? 1 : 0), (Object)(unenhanced != null ? 1 : 0));
        if (enhanced == null && unenhanced == null) {
            throw new NcException("Unable to acquire NetcdfDataset in either enhanced or unenhanced mode.");
        }
        NetcdfDataset njdataset = enhanced == null ? unenhanced : enhanced;
        return new DatasetsStruct(njfile, njdataset, enhanced, unenhanced);
    }

    private static String getGribMessage(String origStr) {
        Object msg = origStr.replaceFirst(".*Could not find a table for GRIB file with ", "No GRIB table available for ");
        if (NcConfig.isGrib1Strict()) {
            msg = (String)msg + ". You might be able to open this file by disabling GRIB-1 strict mode in preferences and re-starting.";
        }
        return msg;
    }

    private static DatasetsStruct acquireFromNcml(DatasetUrl durl) throws IOException {
        NetcdfDataset unenhanced;
        NetcdfDataset enhanced;
        NetcdfFile njfile;
        Exception excx;
        block11: {
            block10: {
                String addStr = durl.getTrueurl();
                LOGGER.trace("{}", (Object)addStr);
                if (durl == null) {
                    LOGGER.warn("Cannot acquire NCML because DatasetUrl is null");
                    LOGGER.trace("Address was {}", (Object)addStr);
                    throw new NcException("Cannot acquire NCML because DatasetUrl is null");
                }
                excx = null;
                njfile = null;
                enhanced = null;
                unenhanced = null;
                try {
                    enhanced = NetcdfDatasets.acquireDataset(durl, ENHANCEMENT, null);
                }
                catch (Exception exc) {
                    excx = exc;
                    LOGGER.trace("Unable to acquire enhanced NCML dataset: {}", (Object)exc.getMessage());
                    if (!LOGGER.isTraceEnabled()) break block10;
                    exc.printStackTrace();
                }
            }
            try {
                unenhanced = NetcdfDatasets.acquireDataset(durl, false, null);
            }
            catch (Exception exc) {
                LOGGER.trace("Unable to acquire unenhanced NCML dataset: {}", (Object)exc.getMessage());
                if (!LOGGER.isTraceEnabled()) break block11;
                exc.printStackTrace();
            }
        }
        LOGGER.trace("njdataset enhanced exists? {}", (Object)(enhanced != null ? 1 : 0));
        LOGGER.trace("njdataset unenhanced exists? {}", (Object)(unenhanced != null ? 1 : 0));
        if (enhanced == null && unenhanced == null) {
            if (excx instanceof FileNotFoundException) {
                throw new NcException("Unable to acquire NCML dataset. A file-not-found exception occurred.");
            }
            if (excx != null) {
                throw new NcException("Unable to acquire NCML dataset due to " + excx.getClass().getSimpleName() + ".");
            }
            throw new NcException("Unable to acquire NCML dataset. No exception was reported; simply got a null object.");
        }
        NetcdfDataset njdataset = enhanced == null ? unenhanced : enhanced;
        try {
            njfile = ((NetcdfDataset.Builder)njdataset.toBuilder()).orgFile;
        }
        catch (Exception exc) {
            LOGGER.debug("Exception getting referenced NetcdfFile: {}", (Object)exc.getMessage());
        }
        LOGGER.trace("njfile exists? {}", (Object)(njfile != null ? 1 : 0));
        return new DatasetsStruct(njfile, njdataset, enhanced, unenhanced);
    }

    private void finishInit() {
        LOGGER.trace("== Convention {}", (Object)this.datasets_.njd_.getConventionUsed());
        String fts = this.getGlobalAttributeString("featureType");
        LOGGER.trace("== explicit featureType {}", (Object)fts);
        if (fts == null) {
            fts = this.getGlobalAttributeString("cdm_data_type");
            LOGGER.trace("== cdm_data_type {}", (Object)fts);
        }
        if (fts == null) {
            LOGGER.trace("== No featureType global attribute");
        } else {
            if ("trajectory".equalsIgnoreCase(fts)) {
                this.geometry_ = NcGeometryType.TRAJECTORY;
            } else if ("trajectoryProfile".equalsIgnoreCase(fts)) {
                this.geometry_ = NcGeometryType.TRAJECTORY_PROFILE;
            } else if ("timeSeries".equalsIgnoreCase(fts)) {
                this.geometry_ = NcGeometryType.TIME_SERIES;
            } else if ("point".equalsIgnoreCase(fts)) {
                this.geometry_ = NcGeometryType.POINT;
            } else if ("profile".equalsIgnoreCase(fts)) {
                this.geometry_ = NcGeometryType.PROFILE;
            } else if ("timeSeriesProfile".equalsIgnoreCase(fts)) {
                this.geometry_ = NcGeometryType.TIME_SERIES_PROFILE;
            }
            LOGGER.trace("== featureType {}", (Object)this.geometry_);
        }
        this.setGrouping(this.datasets_.njd_.getRootGroup());
        this.initChildren();
    }

    public String getAddress() {
        return this.addStr_;
    }

    NetcdfDataset getNjDataset() {
        return this.datasets_.njd_;
    }

    NetcdfDataset getNjDatasetUnenhanced() {
        return this.datasets_.unenhanced_;
    }

    public String getGlobalAttributeString(String name) {
        Attribute a = this.datasets_.njd_.findGlobalAttribute(name);
        if (a == null || a.isArray() || !a.isString()) {
            return null;
        }
        return a.getStringValue();
    }

    public Integer getGlobalAttributeInteger(String name) {
        Attribute a = this.datasets_.njd_.findGlobalAttribute(name);
        if (a == null || a.isArray()) {
            return null;
        }
        Number nn = a.getNumericValue();
        if (nn == null) {
            return null;
        }
        return nn.intValue();
    }

    public NcGeometryType getGeometry() {
        if (this.geometry_ == null) {
            LOGGER.trace("Geometry is null. Returning GRIDDED.");
            this.geometry_ = NcGeometryType.GRIDDED;
        }
        return this.geometry_;
    }

    public Dimension getNjDimension(String fullName) {
        return this.datasets_.njd_.findDimension(fullName);
    }

    Group getNjGroupUnenhanced(String fullName) {
        if (this.datasets_.unenhanced_ == null) {
            return null;
        }
        return this.datasets_.unenhanced_.findGroup(fullName);
    }

    public NcVariable getVariableAtPath(String path) {
        NcDataNode node = this.getChildAtPath(path);
        if (node instanceof NcVariable) {
            return (NcVariable)node;
        }
        if (node == null) {
            LOGGER.warn("No node at path {}.", (Object)path);
        } else {
            LOGGER.warn("Node at path {} is not a variable", (Object)path);
        }
        return null;
    }

    public VariableDS getVariableDS(String fullName) {
        Variable njvar = this.datasets_.njd_.findVariable(fullName);
        if (njvar instanceof VariableDS) {
            return (VariableDS)njvar;
        }
        return null;
    }

    Variable getNjVariableUnenhanced(String fullName) {
        if (this.datasets_.unenhanced_ == null) {
            return null;
        }
        return this.datasets_.unenhanced_.findVariable(fullName);
    }

    public VariableDS getCoordinateVariable(Dimension d) {
        String dname = d.getFullName();
        if (dname == null) {
            return null;
        }
        VariableDS varDS = this.getVariableDS(dname);
        if (varDS != null) {
            return varDS;
        }
        for (TreeTableNode node : this.getChildren()) {
            Dimension d2;
            NcVariable ncv;
            NcDataNode ncn = (NcDataNode)node;
            if (!(ncn instanceof NcVariable) || !(ncv = (NcVariable)ncn).getName().equalsIgnoreCase(dname) || !(d2 = (varDS = (VariableDS)ncv.getObject()).getDimension(0)).equals(d)) continue;
            return varDS;
        }
        return null;
    }

    public void close() throws IOException {
        this.datasets_.close();
    }

    @Override
    public Object getObject() {
        return this.datasets_.njd_;
    }

    @Override
    public String getName() {
        return this.fname_;
    }

    @Override
    public String getNjPath() {
        return this.url_.toString();
    }

    @Override
    public String getShortName() {
        return "";
    }

    @Override
    public String getLongName() {
        Attribute a = this.datasets_.njd_.findGlobalAttributeIgnoreCase("title");
        if (a != null) {
            return a.getStringValue();
        }
        return this.getName();
    }

    @Override
    public String getNcType() {
        String s = this.url_.toString().toLowerCase();
        if (s.startsWith("file:")) {
            return "Local File";
        }
        if (s.startsWith("http:") || s.startsWith("https:") || s.startsWith("ftp:")) {
            return "Remote File";
        }
        if (s.startsWith("dap4:") || s.startsWith("dods:") || s.startsWith("cdms3:") || s.startsWith("s3:")) {
            return "Remote File";
        }
        return "-";
    }

    @Override
    public String getDetail(boolean enhanced) {
        boolean showDSEnhanced = enhanced || this.datasets_.unenhanced_ == null && this.datasets_.njf_ == null;
        StringBuilder sb = new StringBuilder(4096);
        sb.append("<html>");
        sb.append("<head>");
        this.appendDetailStyle(sb);
        sb.append("</head>").append("<body>");
        sb.append("<h2>File \"").append(this.getName()).append("\"</h2>\n");
        if (LOGGER.isDebugEnabled()) {
            sb.append("<hr>").append("<pre>").append("enhanced: ").append(enhanced).append('\n').append("showDSEnhanced: ").append(showDSEnhanced).append("\n\n").append("(unenhanced_ == null): ").append(this.datasets_.unenhanced_ == null).append('\n').append("(njf_ == null): ").append(this.datasets_.njf_ == null).append('\n').append("</pre>");
        }
        sb.append("<p>File type: ");
        if (showDSEnhanced) {
            sb.append(this.datasets_.njd_.getFileTypeDescription());
        } else if (this.datasets_.unenhanced_ == null) {
            sb.append(this.datasets_.njf_.getFileTypeDescription());
        } else {
            sb.append(this.datasets_.unenhanced_.getFileTypeDescription());
        }
        if (LOGGER.isDebugEnabled()) {
            sb.append("<br />").append("File type ID: ");
            if (showDSEnhanced) {
                sb.append(this.datasets_.njd_.getFileTypeId());
            } else if (this.datasets_.unenhanced_ == null) {
                sb.append(this.datasets_.njf_.getFileTypeId());
            } else {
                sb.append(this.datasets_.unenhanced_.getFileTypeId());
            }
            sb.append("</p>\n");
        }
        sb.append("</p>\n");
        if (showDSEnhanced) {
            String convention;
            int lastDot;
            sb.append("<p><b>");
            if (!enhanced) {
                sb.append("Unable to acquire unenhanced dataset. ");
            }
            sb.append("Showing enhanced mode description.</b></p>").append("<p>");
            IOServiceProvider iosp = this.datasets_.njd_.getIosp();
            if (iosp != null) {
                sb.append("IOSP file type: ").append(iosp.getFileTypeDescription()).append("<br>");
                String ss = iosp.getClass().toString();
                if (!LOGGER.isDebugEnabled() && (lastDot = ss.lastIndexOf(46)) > -1) {
                    ss = ss.substring(lastDot + 1);
                }
                sb.append("IOSP class: ").append(ss).append("<br>");
            }
            if ((convention = this.datasets_.njd_.getConventionUsed()) != null && !convention.isEmpty()) {
                if (!LOGGER.isDebugEnabled() && (lastDot = convention.lastIndexOf(46)) > -1) {
                    convention = convention.substring(lastDot + 1);
                }
                sb.append("Convention: ").append(convention).append("<br>");
            }
            if (NcConfig.isDevo()) {
                try {
                    CoordSysBuilderIF factory = CoordSysBuilder.factory(this.datasets_.njd_, null);
                    if (factory != null) {
                        int lastDot2;
                        String ss = factory.getClass().toString();
                        if (!LOGGER.isDebugEnabled() && (lastDot2 = ss.lastIndexOf(46)) > -1) {
                            ss = ss.substring(lastDot2 + 1);
                        }
                        sb.append("Coordinate system builder: ").append(ss).append("<br>");
                    }
                }
                catch (Exception ignore) {
                    LOGGER.debug("Could not extract coordinate system builder");
                }
            }
            sb.append("</p>");
            if (NcConfig.isDevo() && iosp != null) {
                sb.append("<hr>").append("IOSP detail:").append("<br>").append("<pre>").append(iosp.getDetailInfo()).append("</pre>");
            }
        }
        sb.append("<hr>").append("<pre>");
        Object detailString = null;
        detailString = showDSEnhanced ? this.datasets_.njd_.toString() : (this.datasets_.unenhanced_ == null ? this.datasets_.njf_.toString() : this.datasets_.unenhanced_.toString());
        LOGGER.trace("detailString length {}", (Object)((String)detailString).length());
        if (((String)detailString).length() > 300000) {
            LOGGER.info("Detail String is long; truncating.");
            detailString = ((String)detailString).substring(0, 250000) + "\u2026\n\n(truncated to 250,000 chars)";
        }
        sb.append(StringUtils.escapeHtmlSimplistic((String)detailString));
        sb.append("</pre>");
        sb.append("</body>").append("</html>");
        return sb.toString();
    }

    @Override
    public String getDetailType() {
        return "text/html";
    }

    public String getNameRoot() {
        return NcFileUtils.getNameRoot(this.fname_);
    }

    public boolean isHdfEos() {
        for (TreeTableNode node : this.getChildren()) {
            NcGroup ndg;
            NcDataNode ncn = (NcDataNode)node;
            if (!(ncn instanceof NcGroup) || !"HDFEOS".equalsIgnoreCase((ndg = (NcGroup)ncn).getName()) && !"HDFEOS_INFORMATION".equalsIgnoreCase(ndg.getName())) continue;
            return true;
        }
        Attribute a = this.datasets_.njd_.findGlobalAttribute("HDFEOSVersion");
        return a != null && a.isString();
    }

    public boolean isAquaModis() {
        String pstr = this.getGlobalAttributeString("platform");
        String istr = this.getGlobalAttributeString("instrument");
        return "Aqua".equals(pstr) && "MODIS".equals(istr);
    }

    static {
        Set<NetcdfDataset.Enhance> dfltMode = NetcdfDataset.getDefaultEnhanceMode();
        ENHANCEMENT = EnumSet.copyOf(dfltMode);
        ENHANCEMENT.add(NetcdfDataset.Enhance.IncompleteCoordSystems);
    }

    private static final class DatasetsStruct {
        private final NetcdfDataset njd_;
        private final NetcdfDataset enhanced_;
        private final NetcdfDataset unenhanced_;
        private final NetcdfFile njf_;

        DatasetsStruct(NetcdfFile njfile, NetcdfDataset njdataset, NetcdfDataset enhanced, NetcdfDataset unenhanced) {
            this.njf_ = njfile;
            this.njd_ = njdataset;
            this.enhanced_ = enhanced;
            this.unenhanced_ = unenhanced;
        }

        void close() throws IOException {
            if (this.enhanced_ != null) {
                this.enhanced_.close();
            }
            if (this.unenhanced_ != null) {
                this.unenhanced_.close();
            }
            if (this.njf_ != null) {
                this.njf_.close();
            }
        }
    }

    private static enum SourceType {
        FILE,
        URL,
        DATASET_URL,
        STRING;

    }
}

